;==========================================================================
;Version history:
;==========================================================================
;2010/01/14     First edition
;2010/05/31     Rename the "D_MusicEvent" as "D_UserEvent"
;               Modify the subroutine "F_GetUserEvent"; clear "D_UserEvent" flag if an event is detected
;               Add new function "MIDI_EVENT"; use MIDI CH16 for MIDI events
;               Add new subroutine "F_GetMIDIEvent" to get a MIDI event
;2010/07/28     Debug the subroutine "F_SetMIDIIrqPara" for index error
;               Debug the subroutine "F_MIDIOff" for "D_DrumPlaying" judgement
;2010/10/12     Debug the subroutine "F_CheckMIDIChVol" for branch error
;2010/12/01     Check if any single note/drum is played before turning off DAC in "F_StopMusic"
;2011/01/12     Debug "F_StopSound", fix index error for stopping single note/drum
;2011/01/20     Debug the dynamic allocation to avoid disabling the channel which is paused in playing speech
;2011/02/10     Revise the dynamic allocation; stop enabled channel to prevent the loop region error of original instrument/drum
;2011/03/29     Revise the dynamic allocation; skip to allocate the channel which in playing a single note/drum
;2011/03/30     Add new function of IO event to "L_UserEvent" (controlled by "MIDI_IOEvt" in MIDI.inc)
;==========================================================================
        .EXTERN         T_InstrumentsL  ;Declared in library which is generated by G+Midiar
        .EXTERN         T_InstrumentsH
        .EXTERN         T_DrumInstL
        .EXTERN         T_DrumInstH

        .EXTERN         T_SampleRate
        .EXTERN         T_SampleRateBase
        .EXTERN         D_PhaseIndexOffset
;==========================================================================
        .INCLUDE        Inc\MIDI.inc
        .INCLUDE        Tab\MIDI.tab
;==========================================================================
        .CODE
;==========================================================================
;Purpose: Play a MIDI
;Input: X = MIDI index, Y = Dynamic Allocation Type
;Dynamic Type: 0 = Dynamic Off, 1 = Oldest note out, 2 = Newest note out, 3 = Minimum volume note out
;Destroy: A, X, Y
;==========================================================================
F_PlayMIDI:                             ;X = MIDI index
        LDA     R_MIDIStatus
        AND     #D_MusicOff
        BEQ     L_PlayM?
        JMP     L_ExitPlayMIDI
L_PlayM?:
        LDA     T_MIDI_L,X
        STA     R_MIDIDPTR              ;Low byte

        LDA     T_MIDI_H,X
        STA     R_MIDIDPTR+1            ;High byte

        LDA     T_MIDI_B,X
        CMP     #03H
        BCS     L_SetBank?
        LDA     #03H
L_SetBank?:
        STA     R_MIDIDPTR+2            ;Bank

        .IF REPEAT_MIDI = ON
        BIT     R_RepeatMIDI            ;Bit7 is a flag of MIDI repeat mode
        BMI     L_KeepOnRepeat?
L_NoRepeat?:
        STX     R_RepeatMIDI
        JMP     L_GoOn?
L_KeepOnRepeat?:
        TXA                             ;MIDI index
        ORA     #D_RepeatMIDI           ;Enable MIDI repeat mode
        STA     R_RepeatMIDI            ;Update MIDI index
L_GoOn?:
        .ENDIF

        LDA     #D_TempoIndexOffset     ;Set default tempo for adjustable
        STA     R_TempoIndex

        LDA     P_DAC_Ctrl
        AND     #(D_DAC_Enable+D_OPN_En+D_OPP_En)
        CMP     #(D_DAC_Enable+D_OPN_En+D_OPP_En)
        BEQ     L_GOM?
        %TurnOnDAC
L_GOM?:
        SEI
        LDA     #00H                    ;Initialize delta time
        STA     R_DeltaTime
        STA     R_DeltaTime+1
        CLI

        STA     R_MainIndex             ;Used for user event
        STA     R_SubIndex

        .IF MIDI_EVENT = ON
        STA     R_MIDIEvent
        .ENDIF

        .IF MUTE_MIDI_CH = ON
        STA     R_EnCh1ToCh8
        STA     R_EnCh9ToCh16
        .ENDIF

        .IF MIDI_KEY_SHIFT = ON
        STA     R_KeyShift
        .ENDIF

        .IF CTRL_MIDI_CH_VOL = ON
        STA     R_CtrlVolCh1ToCh8
        STA     R_CtrlVolCh9ToCh16
        LDA     #D_DefaultVolumeLevel
        STA     R_CtrlMIDIChVol         ;Set default control volume
        .ENDIF
 
        LDA     R_MIDIStatus
        AND     #.NOT.(D_DeltaTime+D_DynaAllocOff+D_NewestNoteOut+D_MinVolNoteOut+D_UserEvent)
        ORA     #D_MIDIEn               ;Set flag
        STA     R_MIDIStatus

        LDA     R_MIDICtrl
        AND     #.NOT.(D_PauseMIDI+D_MIDIEvent)
        STA     R_MIDICtrl

        LDX     #00H                    ;Initial for playing a MIDI
L_Initial:
        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying
        BNE     L_Next?
        .ENDIF

        LDA     R_SingleNote            ;Check if a single note or drum is played in this channel
        ORA     R_SingleDrum
        AND     T_ChEnable,X
        BNE     L_Next?

        SEI
        LDA     P_SPU_Enable
        AND     T_ChDisable,X
        STA     P_SPU_Enable

        LDA     #00H
        STA     R_ChTime,X

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,X
        STA     P_SPU_IntCtrl
        CLI
L_Next?:
        INX
        CPX     #D_ChannelNo
        BCC     L_Initial

        LDX     #00H                    ;Set default instrument
L_SetDefaultInst:
        LDA     #00H                    ;Default instrument is index 0
        STA     R_ChInst,X
        INX
        CPX     #10H                    ;Set default instrument for 16 channels 
        BCC     L_SetDefaultInst
L_ExitSetDefaultInst:
        LDX     #00H
L_DurationClr?:
        STA     R_MidiDurationL,X       ;Clear duration
        STA     R_MidiDurationH,X
        INX
        CPX     #D_ChannelNo
        BCC     L_DurationClr?

        JSR     F_SelectDynamic

        SEI
        LDA     P_INT_CtrlL
        ORA     #D_TB1024IntEn
        STA     P_INT_CtrlL

        LDA     P_INT_CtrlH
        ORA     #D_SPUIntEn
        STA     P_INT_CtrlH
        CLI
L_ExitPlayMIDI:
        RTS
;==========================================================================
F_CheckMidiDuration:
        LDA     R_MIDIStatus
        AND     #D_MusicOff
        BEQ     L_PlayM?
        RTS

L_PlayM?:
        LDA     R_MIDIStatus
        BMI     L_MIDIPlaying
L_ExitMidiDuraCheck:
        RTS

L_MIDIPlaying:                          ;If MidiDuration = 0, set it as note off
        LDX     #D_ChannelNo-1
L_CheckDurationLoop:
        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying
        BNE     L_CheckNextDuration?
        .ENDIF

        SEI
        LDA     R_MidiDurationL,X
        ORA     R_MidiDurationH,X
        CLI
        BNE     L_CheckNextDuration?

        LDA     T_ChEnable,X
        BIT     R_SingleNote            ;If a single note is played, don't check duration
        BNE     L_CheckNextDuration?

        BIT     R_SingleDrum            ;If a single drum is played, don't check duration
        BNE     L_CheckNextDuration?

        SEI
        LDA     R_ChStatus,X
        ORA     #D_NoteOffFlag
        STA     R_ChStatus,X            ;Set Note off flag
        CLI
        LDA     #00H
        STA     R_VolumeFloat,X         ;Initialize float part of channel volume to calculate new release value
L_CheckNextDuration?:
        DEX
        BPL     L_CheckDurationLoop
        RTS
;==========================================================================
F_ServiceMIDI:
        LDA     R_MIDICtrl
        AND     #D_PauseMIDI
        BEQ     L_PlayM1?
        JMP     L_ExitServiceMIDI

L_PlayM1?:
        LDA     R_MIDIStatus
        AND     #D_MusicOff
        BEQ     L_PlayM?
        JMP     L_ExitServiceMIDI

L_PlayM?:
        LDA     R_MIDIStatus
        BMI     L_MIDINormalPlay
L_ExitServiceMIDI:
        RTS

L_MIDINormalPlay:
        SEI
        LDA     R_DeltaTime             ;Decreased in Fcpu/1024 IRQ
        ORA     R_DeltaTime+1
        CLI
        BNE     L_ExitServiceMIDI
L_DataFetchLoop:
        LDX     #D_ChannelNo
L_DecChTimeLoop?:
        LDA     R_ChTime-1,X
        BEQ     L_ChkNextChTime?
        DEC     R_ChTime-1,X
L_ChkNextChTime?:
        DEX
        BNE     L_DecChTimeLoop?

        JSR     F_GetMIDIData           ;Get MIDI data
        LDA     R_MIDIStatus
        AND     #D_DeltaTime
        BEQ     L_IsDeltaTime
L_IsNotDeltaTime:
        LDA     R_MIDIData              ;It's not Delta time
        AND     #00001111B              ;Bit0~3 = Event
        CMP     #D_ProcessEventSize     ;Number of event
        BCC     L_ContinueJmp
        RTS

L_IsDeltaTime:
        LDA     R_MIDIStatus            ;Delta time flag
        ORA     #D_DeltaTime
        STA     R_MIDIStatus
        SEI
        LDA     R_MIDIData
        STA     R_DeltaTime
        LDA     #00H
        STA     R_DeltaTime+1
        CLI
        RTS

L_ContinueJmp:
        TAX                             ;Bit0~3 of data is event number
        LDA     T_ProcessDataEventL,X
        STA     R_InstAddr
        LDA     T_ProcessDataEventH,X
        STA     R_InstAddr+1
        JMP     (R_InstAddr)

T_ProcessDataEventL:
        DB      .LOW.(L_ProcessNote)
        DB      .LOW.(L_ProcessTempo)
        DB      .LOW.(L_ProcessInst)
        DB      .LOW.(L_EndedCode)
        DB      .LOW.(L_LongDeltaTime)
        DB      .LOW.(L_PitchBend)
        DB      .LOW.(L_ADSRTimeBase)
        DB      .LOW.(L_NoteOffTimeBase)
        DB      .LOW.(L_UserEvent)
D_ProcessEventSize:     EQU     $-T_ProcessDataEventL

T_ProcessDataEventH:
        DB      .HIGH.(L_ProcessNote)
        DB      .HIGH.(L_ProcessTempo)
        DB      .HIGH.(L_ProcessInst)
        DB      .HIGH.(L_EndedCode)
        DB      .HIGH.(L_LongDeltaTime)
        DB      .HIGH.(L_PitchBend)
        DB      .HIGH.(L_ADSRTimeBase)
        DB      .HIGH.(L_NoteOffTimeBase)
        DB      .HIGH.(L_UserEvent)
;==========================================================================
;Event 0
;==========================================================================
;    8-bit        4-bit  4-bit        8-bit         8-bit         8-bit         8-bit
; Delta-time     Channel Event     Note Index  Velocity*Volume Duration_LB   Duration_HB
;==========================================================================
L_ProcessNote:
        LDA     R_MIDIData
        LSR     A
        LSR     A
        LSR     A
        LSR     A
        STA     R_MIDIData              ;Get MIDI channel

        LDA     R_MIDIStatus
        AND     #D_DynaAllocOff
        BNE     L_DynamicAllocOff

        LDA     R_MIDIStatus
        AND     #D_NewestNoteOut
        BNE     L_NewestNoteOut

        LDA     R_MIDIStatus
        AND     #D_MinVolNoteOut
        BEQ     L_OldestNoteOut
L_MinVolNoteOut:
        JSR     F_FindMinChVol          ;Dynamic allocation with minimum volume note out
        JMP     L_DynamicAllocOn
L_NewestNoteOut:
        JSR     F_FindNewestNote        ;Dynamic allocation with newest note out
        JMP     L_DynamicAllocOn
L_OldestNoteOut:
        JSR     F_FindOldestNote        ;Dynamic allocation with oldest note out
        JMP     L_DynamicAllocOn
;==========================================================================
;Dynamic off
;==========================================================================
L_DynamicAllocOff:                      ;No dynamic allocation
        LDA     R_MIDIData              ;MIDI channel
        CMP     #09H                    ;Check if it's the CH10 (Drum)
        BEQ     L_MapToHWCh
        STA     R_BackUpCh
        JMP     L_AllocatedCh
L_MapToHWCh:
        LDX     #D_ChannelNo-1          ;Map MIDI channel 10 to last physical channel when dynamic off
        STX     R_BackUpCh
        JMP     L_AllocatedCh
;==========================================================================
;Dynamic allocation
;==========================================================================
L_DynamicAllocOn:
        LDX     R_BackUpCh
        LDA     #FFH
        STA     R_ChTime,X              ;Initialize channel time

        LDA     R_MIDIData              ;MIDI channel
L_AllocatedCh:
        .IF PITCH_BEND = ON
        LDX     R_BackUpCh
        STA     R_SpuMidiCh,X
        .ENDIF

        .IF MUTE_MIDI_CH = ON
        TAX
        LDA     T_Table16L,X
        AND     R_EnCh1ToCh8
        BEQ     L_Check9To16?
        JMP     L_DisableMIDIChannel
L_Check9To16?:
        LDA     T_Table16H,X
        AND     R_EnCh9ToCh16
        BEQ     L_ExitChkMIDICh
L_DisableMIDIChannel:
        %ClearDeltaTimeFlag
        JSR     F_GetMIDIData
        JSR     F_GetMIDIData
        LDA     R_MIDIData              ;Process Velocity X Volume
        BMI     L_TwoByteDurationDisable
        JSR     F_GetMIDIData
        JMP     L_ExitDisableMIDIChannel
L_TwoByteDurationDisable:
        JSR     F_GetMIDIData
        JSR     F_GetMIDIData
L_ExitDisableMIDIChannel:
        RTS
L_ExitChkMIDICh:
        .ENDIF

        LDX     R_BackUpCh
        SEI
        LDA     R_ChStatus,X
        AND     #.NOT.(D_NoteOffFlag)
        STA     R_ChStatus,X
        CLI

        LDA     R_MIDIData              ;MIDI channel
        CMP     #09H                    ;MIDI CH10 = Drum
        BNE     L_MIDIEvent
        JMP     L_GetDrum
;==========================================================================
;Deal with MIDI Event
;==========================================================================
L_MIDIEvent:
        .IF MIDI_EVENT = ON
        CMP     #0FH                    ;MIDI CH16 = MIDI Event if MIDI_EVENT is "ON"
        BEQ     L_GetMIDIEvent?
        JMP     L_GetNote
L_GetMIDIEvent?:
        LDA     R_MIDICtrl              ;Set flag
        ORA     #D_MIDIEvent
        STA     R_MIDICtrl
L_GetPitch?:
        JSR     F_GetMIDIData           ;Get pitch
        LDA     R_MIDIData
        STA     R_MIDIEvent
L_GetVolume?
        JSR     F_GetMIDIData           ;Get volume
        LDA     R_MIDIData
        BMI     L_TwoBytesDuration?
L_OneByteDuration?
        JSR     F_GetMIDIData           ;Get one byte duration
        JMP     L_ExitMIDIEvent?
L_TwoBytesDuration?:
        JSR     F_GetMIDIData           ;Get two bytes duration
        JSR     F_GetMIDIData
L_ExitMIDIEvent?:
        %ClearDeltaTimeFlag
        RTS
        .ENDIF
;==========================================================================
L_GetNote:                              ;Get a note
        LDX     R_BackUpCh              ;A = MIDI Channel
        STA     R_InstIndex,X           ;Backup MIDI channel number of instrument

        LDA     R_MIDIStatus
        AND     #.NOT.(D_UserEvent)
        STA     R_MIDIStatus

        .IF MIDI_EVENT = ON
        LDA     R_MIDICtrl
        AND     #.NOT.(D_MIDIEvent)
        STA     R_MIDICtrl
        .ENDIF

        JSR     F_GetMIDIData
        LDX     R_BackUpCh
        LDA     R_MIDIData              ;Note pitch
L_StoreNote:
        .IF MIDI_KEY_SHIFT = ON
        CLC
        ADC     R_KeyShift
        .ENDIF

        STA     R_Note,X
        JSR     F_GetMIDIData
        LDA     R_MIDIData              ;Process Velocity X Volume
        BMI     L_TwoByteDuration       ;Bit7 = 0, 1 Byte Duration
L_OneByteDuration:
        BNE     L_GO?
        LDA     #7FH                    ;Set Velocity X Volume as default value
L_GO?:
        LDX     R_BackUpCh
        STA     R_VeloVol,X

        JSR     F_GetMIDIData
        LDX     R_BackUpCh
        SEI
        LDA     R_MIDIData
        STA     R_MidiDurationL,X       ;Process duration
        LDA     #00H
        STA     R_MidiDurationH,X
        CLI
        JMP     L_PlayNote

L_TwoByteDuration:
        AND     #01111111B              ;Process Velocity X Volume
        BNE     L_GO?
        LDA     #7FH                    ;Set Velocity X Volume as default value
L_GO?:
        LDX     R_BackUpCh
        STA     R_VeloVol,X

        JSR     F_GetMIDIData
        LDA     R_MIDIData              ;Backup duration high byte
        STA     R_MIDITemp

        JSR     F_GetMIDIData
        LDX     R_BackUpCh
        SEI
        LDA     R_MIDIData              ;Process duration
        STA     R_MidiDurationH,X       ;Duration high byte
        LDA     R_MIDITemp
        STA     R_MidiDurationL,X       ;Duration low byte
        CLI

L_PlayNote:
        %ClearDeltaTimeFlag             ;Clear flag of Delta time

        .IF CTRL_MIDI_CH_VOL = ON
        JSR     F_CheckMIDIChVol
        .ENDIF

        JSR     F_InitialADSR
        JMP     F_ChPlayTone            ;Play note
;==========================================================================
L_GetDrum:                              ;Get a drum
        LDA     R_MIDIStatus
        AND     #.NOT.(D_UserEvent)
        STA     R_MIDIStatus

        .IF MIDI_EVENT = ON
        LDA     R_MIDICtrl
        AND     #.NOT.(D_MIDIEvent)
        STA     R_MIDICtrl
        .ENDIF

        JSR     F_GetMIDIData
        LDA     R_MIDIData
        STA     R_DrumIndex             ;Backup drum index

        JSR     F_GetMIDIData
        LDA     R_MIDIData              ;Process Velocity X Volume
        BMI     L_DrumTwoByteDuration
L_DrumOneByteDuration:
        BNE     L_GO?
        LDA     #7FH                    ;Set Velocity X Volume as default value
L_GO?
        LDX     R_BackUpCh
        STA     R_VeloVol,X
        JSR     F_GetMIDIData
        LDX     R_BackUpCh
        SEI
        LDA     R_MIDIData
        STA     R_MidiDurationL,X       ;Duration of drum
        LDA     #00H
        STA     R_MidiDurationH,X       ;Duration of drum
        CLI
        JMP     L_PlayDrum

L_DrumTwoByteDuration:
        AND     #01111111B              ;Process Velocity X Volume(Bit7 = 0, 1 Byte duration)
        BNE     L_GO?
        LDA     #7FH                    ;Set Velocity X Volume as default value
L_GO?
        LDX     R_BackUpCh
        STA     R_VeloVol,X
        JSR     F_GetMIDIData
        LDA     R_MIDIData
        STA     R_MIDITemp

        JSR     F_GetMIDIData
        LDX     R_BackUpCh
        SEI
        LDA     R_MIDIData              ;Process duration of drum
        STA     R_MidiDurationH,X
        LDA     R_MIDITemp
        STA     R_MidiDurationL,X
        CLI

L_PlayDrum:
        %ClearDeltaTimeFlag             ;Clear flag of Delta time

        .IF CTRL_MIDI_CH_VOL = ON
        LDA     R_CtrlVolCh9ToCh16
        AND     #D_CtrlCh10Vol
        BEQ     L_ExitCheckCh10Vol

        LDA     R_VeloVol,X
        STA     R_FaciendL
        LDX     R_CtrlMIDIChVol
        LDA     T_VolumeLevel,X
        STA     R_MultiplierL
        INX
        LDA     T_VolumeLevel,X
        STA     R_MultiplierM
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        BNE     L_SetToMaxDrum?
        LDA     R_ProductM
        CMP     #80H
        BCC     L_SetVolumeDrum?
L_SetToMaxDrum?:
        LDA     #7FH
L_SetVolumeDrum?:
        LDX     R_BackUpCh
        STA     R_VeloVol,X
L_ExitCheckCh10Vol:
        .ENDIF

        JSR     F_InitialDrumADSR
        JMP     F_PlayDrum
;==========================================================================
;Event 1
;==========================================================================
L_ProcessTempo:                         ;If don't change the tempo, not use it 
        %ClearDeltaTimeFlag
        LDA     R_MIDIData
        AND     #11110000B
        LSR     A
        LSR     A
        LSR     A
        LSR     A
        JSR     F_GetMIDIData
        LDA     R_MIDIData
L_ExitProcessTempo:
        RTS
;==========================================================================
;Event 2
;==========================================================================
L_ProcessInst:                          ;Set the instrument index of MIDI channel
        %ClearDeltaTimeFlag
        LDA     R_MIDIData
        AND     #11110000B
        LSR     A
        LSR     A
        LSR     A
        LSR     A
        TAX
        JSR     F_GetMIDIData
        LDA     R_MIDIData
        STA     R_ChInst,X
L_ExitProcessInst:
        RTS
;==========================================================================
;Event 3
;==========================================================================
L_EndedCode:                            ;End code of a MIDI
F_StopMusic:                            ;Stop MIDI
        LDY     #00H
L_StopSoundLoop:                        ;Stop channel 0,1,2,3 sound
        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,Y
        AND     #D_SpeechPlaying
        BNE     L_ChkNextChStop         ;If Speech is playing, don't stop sound
        .ENDIF

        LDA     T_ChEnable,Y
        BIT     R_SingleNote            ;Check if a single note is played in this channel
        BNE     L_ChkNextChStop

        BIT     R_SingleDrum            ;Check if a single drum is played in this channel
        BNE     L_ChkNextChStop

        SEI
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_NotePlaying+D_NoteOffFlag+D_TargetIsLastDrop+D_ReachLastDrop+D_DrumPlaying+D_AdpcmJumpPcm+D_ZCJump+D_SkipIniNoteOffEnv)
        STA     R_ChStatus,Y            ;Clear SPU IRQ flag and MIDI playing Flag
        CLI
        JSR     F_StopSound
L_ChkNextChStop:
        INY
        CPY     #D_ChannelNo
        BCC     L_StopSoundLoop

        LDA     R_MIDIStatus
        AND     #.NOT.(D_MIDIEn+D_UserEvent+D_DeltaTime)
        STA     R_MIDIStatus

        LDA     R_MIDICtrl
        AND     #.NOT.(D_PauseMIDI+D_MIDIEvent)
        STA     R_MIDICtrl

        .IF REPEAT_MIDI = ON
        BIT     R_RepeatMIDI
        BMI     L_RepeatMIDIOn?
        JMP     L_RepeatMIDIOff?
L_RepeatMIDIOn?:
        LDA     R_RepeatMIDI
        AND     #~D_Bit7
        TAX
        JMP     F_PlayMIDI
        .ENDIF

L_RepeatMIDIOff?:
        .IF SPEECH = ON
        LDA     R_SpeechStatus          ;Check if any speech is played with SPU
        BNE     L_ExitStopLoop
        .ENDIF

        .IF SW_ChA = ON
        %IsActiveChA                    ;Check if SW_ChA is active
        BCS     L_ExitStopLoop
        .ENDIF

        .IF SW_ChB = ON
        %IsActiveChB                    ;Check if SW_ChB is active
        BCS     L_ExitStopLoop
        .ENDIF

        .IF ComAir = ON
        JSR     F_CA_IsActive
        BCS     L_ExitStopLoop
        .ENDIF

        LDA     R_SingleNote            ;Check if any single note is played
        ORA     R_SingleDrum            ;Check if any single drum is played
        BNE     L_ExitStopLoop

        %TurnOffDAC

L_ExitStopLoop:
        RTS
;==========================================================================
;Event 4
;==========================================================================
L_LongDeltaTime:                        ;Long delta time
        JSR     F_GetMIDIData
        LDX     R_MIDIData
        JSR     F_GetMIDIData
        LDA     R_MIDIData
        SEI
        STA     R_DeltaTime
        STX     R_DeltaTime+1
        CLI
        RTS
;==========================================================================
;Event 5
;==========================================================================
L_PitchBend:                            ;Process pitch bend
        .IF PITCH_BEND = ON
        %ClearDeltaTimeFlag
        LDA     R_MIDIData
        AND     #11110000B
        LSR     A
        LSR     A
        LSR     A
        LSR     A                       ;A = MIDI Pitch Bend CH.
        STA     R_PitchBendMidiCh

        JSR     F_GetMIDIData
        LDA     R_MIDIData              ;R_MIDIData = Pitch Bend Level
        STA     R_PitchBendLevel

        LDA     R_PitchBendLevel
        AND     #00111111B
        STA     R_PitchBendLevel
        CMP     #04H
        BCC     L_ExitPitchBend         ;If R_PitchBendLevel<=3, skip pitch bend

        LDX     #00H
L_CheckPitchBendCh:
        LDA     R_SpuMidiCh,X
        CMP     R_PitchBendMidiCh
        BEQ     L_GetChNoteFs           ;If MIDI channel = Pitch Bend channel, deal with pitch bend
L_CheckNext:
        INX
        CPX     #D_ChannelNo
        BCS     L_ExitPitchBend
        JMP     L_CheckPitchBendCh
L_GetChNoteFs:
        STX     R_MIDITemp              ;Backup the SPU channel of pitch bend
        JSR     F_TonePitchBend
        LDX     R_MIDITemp              ;Restore the SPU channel of pitch bend
        JMP     L_CheckNext             ;Check multi notes in one MIDI pitch bend channel
L_ExitPitchBend:
        RTS
        .ELSE
        %ClearDeltaTimeFlag
        JSR     F_GetMIDIData
        RTS
        .ENDIF
;==========================================================================
;Event 6
;==========================================================================
L_ADSRTimeBase:                         ;Set the timebase of envelope
        %ClearDeltaTimeFlag
        JSR     F_GetMIDIData
        LDA     R_MIDIData
        STA     R_FaciendL
        LDA     #<D_TimeBaseFactor
        STA     R_MultiplierL
        LDA     #>D_TimeBaseFactor
        STA     R_MultiplierM
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM  ;ADSR Timebase is processed in IRQ 1024 X 4
        LDA     R_ProductM
        STA     R_ADSRTimeBase
        RTS
;==========================================================================
;Event 7
;==========================================================================
L_NoteOffTimeBase:                      ;Set the timebase of note off
        %ClearDeltaTimeFlag
        JSR     F_GetMIDIData
        LDA     R_MIDIData
        STA     R_FaciendL
        LDA     #<D_TimeBaseFactor
        STA     R_MultiplierL
        LDA     #>D_TimeBaseFactor
        STA     R_MultiplierM
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM  ;TimeBase (Note Off Timebase is processed in IRQ 1024 X 4
        LDA     R_ProductM              ;NoteOff_TimeBase is the time user defined in SunMidiar2
        STA     R_NoteOffTimeBase
        RTS
;==========================================================================
;Event 8
;==========================================================================
L_UserEvent:                            ;Process user events of a MIDI (NRPN)
        %ClearDeltaTimeFlag
        JSR     F_GetMIDIData
        LDA     R_MIDIData
        STA     R_MainIndex             ;Store main index of user event

        JSR     F_GetMIDIData
        LDA     R_MIDIData
        STA     R_SubIndex              ;Store sub index of user event

        .IF MIDI_IOEvt = ON
        LDA     R_MainIndex
        CMP     #80H
        BCC     L_MIDI_UserEvent?
        CMP     #A0H
        BCC     L_MIDI_IOEvent?
        CMP     #F0H
        BEQ     L_MIDI_IOEventStart?
        CMP     #F1H
        BEQ     L_MIDI_IOEventEnd?
        RTS

L_MIDI_UserEvent?:
        LDA     R_MIDIStatus            ;Set flag for user event
        ORA     #D_UserEvent
        STA     R_MIDIStatus
        RTS
L_MIDI_IOEvent?:
        LDY     R_SubIndex
        JSR     F_IOEventProcess        ;Start a PWMIO
        RTS
L_MIDI_IOEventStart?:
        JSR     F_IOEventEnd            ;Set all PWMIOs as output low
        RTS
L_MIDI_IOEventEnd?:
        JSR     F_IOEventEnd            ;Set all PWMIOs as output low
        RTS
        .ELSE
        LDA     R_MIDIStatus
        ORA     #D_UserEvent
        STA     R_MIDIStatus
        RTS
        .ENDIF
;==========================================================================
;Target and slope both need to multiply the value of "R_Velovol".
;==========================================================================
F_InitialADSR:                          ;X = Channel number
        LDY     R_BackUpCh
L_IniAdsrLoop:
        LDA     #00H                    ;Initialize offset
        STA     R_ADSROffset
        STA     R_TabOffset,Y
        STA     R_EnvTimeBase,Y

        JSR     F_GetAdsrValue          ;Input = R_ADSROffset; Output = Acc = Integer part of first Target
        ASL     A
        STA     P_MulF                  ;Faciend
        LDY     R_BackUpCh
        LDA     R_VeloVol,Y
        STA     P_MulM                  ;Multiplier
        LDA     #D_MulAct
        STA     P_MulAct
L_Wait?:
        BIT     P_MulAct
        BNE     L_Wait?

        LDA     P_MulOutH
        STA     R_VolumeInteger,Y
        LDA     P_MulOutL
        STA     R_VolumeFloat,Y

        LDY     #01H
        LDA     (R_InstAddr),Y          ;Get integer of slope
        BMI     L_IniSlopeMinus
L_IniSlopePlus:                         ;Jump here if slope is plus
        STA     R_MultiplierM           ;Save integer of slope
        INY
        LDA     (R_InstAddr),Y
        STA     R_MultiplierL           ;Save float of slope
        LDY     R_BackUpCh
        LDA     R_VeloVol,Y
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        ORA     R_ProductM
        BNE     L_Branch?
        LDA     #01H                    ;If slop is zero, set "01H" as minimum value
        STA     R_ProductM
L_Branch?:
        LDA     R_ProductH
        STA     R_SlopeInteger,Y
        LDA     R_ProductM
        STA     R_SlopeFloat,Y
        JMP     L_GetInitialTarget

L_IniSlopeMinus:                        ;Jump here if slope is minus
        EOR     #FFH                    ;Get 2's complement to be plus
        STA     R_MultiplierM           ;Save integer of slope
        INY
        LDA     (R_InstAddr),Y
        EOR     #FFH                    ;Get 2's complement to be plus
        CLC
        ADC     #01H
        STA     R_MultiplierL           ;Save float of slope
        LDY     R_BackUpCh
        LDA     R_VeloVol,Y
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        ORA     R_ProductM
        BNE     L_Branch?
        LDA     #01H                    ;If slop is zero, set "01H" as minimum value
        STA     R_ProductM
L_Branch?:
        LDA     R_ProductH
        EOR     #FFH                    ;Get 2's complement to returm to minus
        STA     R_SlopeInteger,Y
        LDA     R_ProductM
        EOR     #FFH                    ;Get 2's complement to returm to minus
        CLC
        ADC     #01H
        STA     R_SlopeFloat,Y
L_GetInitialTarget:
        LDY     #03H
        LDA     (R_InstAddr),Y          ;Get the target value
        ASL     A
        STA     P_MulF
        LDY     R_BackUpCh              ;Get channel number
        LDA     R_VeloVol,Y
        STA     P_MulM                  ;Multiplier
        LDA     #D_MulAct
        STA     P_MulAct
L_Wait?:
        BIT     P_MulAct
        BNE     L_Wait?

        LDA     P_MulOutH
        STA     R_TargetInteger,Y       ;First integer part of target
        LDA     P_MulOutL
        STA     R_TargetFloat,Y

        SEI
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_TargetIsLastDrop+D_ReachLastDrop+D_DrumPlaying+D_NoteOffFlag+D_SkipIniNoteOffEnv+D_AdpcmJumpPcm+D_ZCJump)
        STA     R_ChStatus,Y
        CLI
        RTS
;==========================================================================
F_InitialDrumADSR:                      ;X = Channel number
        LDY     R_BackUpCh
L_IniDrumAdsrLoop:
        LDA     #00H                    ;Offset (first)
        STA     R_ADSROffset
        STA     R_TabOffset,Y           ;Initialize table offset
        STA     R_EnvTimeBase,Y

        LDY     R_DrumIndex
        JSR     F_GetDrumAdsrValue
        ASL     A
        STA     P_MulF                  ;Faciend
        LDY     R_BackUpCh
        LDA     R_VeloVol,Y
        STA     P_MulM                  ;Multiplier
        LDA     #D_MulAct
        STA     P_MulAct
L_Wait?:
        BIT     P_MulAct
        BNE     L_Wait?

        LDA     P_MulOutH
        STA     R_VolumeInteger,Y
        LDA     P_MulOutL
        STA     R_VolumeFloat,Y

        LDY     #01H
        LDA     (R_InstAddr),Y          ;Get integer of slope
        BMI     L_IniDrumSlopeMinus
L_IniDrumSlopePlus:
        STA     R_MultiplierM           ;Save integer of slope
        INY
        LDA     (R_InstAddr),Y
        STA     R_MultiplierL           ;Get float of slope
        LDY     R_BackUpCh
        LDA     R_VeloVol,Y
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        ORA     R_ProductM
        BNE     L_Branch?
        LDA     #01H                    ;If slop is zero, set "01H" as minimum value
        STA     R_ProductM
L_Branch?:
        LDA     R_ProductH
        STA     R_SlopeInteger,Y
        LDA     R_ProductM
        STA     R_SlopeFloat,Y
        JMP     L_GetDrumInitialTarget

L_IniDrumSlopeMinus:
        EOR     #FFH                    ;Get 2's complement to be plus
        STA     R_MultiplierM           ;Save integer of slope
        INY
        LDA     (R_InstAddr),Y
        EOR     #FFH                    ;Get 2's complement to be plus
        CLC
        ADC     #01H
        STA     R_MultiplierL           ;Save float of slope
        LDY     R_BackUpCh
        LDA     R_VeloVol,Y
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        ORA     R_ProductM
        BNE     L_Branch?
        LDA     #01H                    ;If slop is zero, set "01H" as minimum value
        STA     R_ProductM
L_Branch?:
        LDA     R_ProductH
        EOR     #FFH                    ;Get 2's complement to returm to minus
        STA     R_SlopeInteger,Y
        LDA     R_ProductM
        EOR     #FFH                    ;Get 2's complement to returm to minus
        CLC
        ADC     #01H
        STA     R_SlopeFloat,Y
L_GetDrumInitialTarget:
        LDY     #03H
        LDA     (R_InstAddr),Y          ;Get the target value
        ASL     A
        STA     P_MulF
        LDY     R_BackUpCh              ;Get channel number
        LDA     R_VeloVol,Y
        STA     P_MulM
        LDA     #D_MulAct
        STA     P_MulAct
L_Wait?:
        BIT     P_MulAct
        BNE     L_Wait?

        LDA     P_MulOutH
        STA     R_TargetInteger,Y       ;First integer part of target
        LDA     P_MulOutL
        STA     R_TargetFloat,Y

        SEI
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_TargetIsLastDrop+D_ReachLastDrop+D_NotePlaying+D_NoteOffFlag+D_SkipIniNoteOffEnv+D_AdpcmJumpPcm+D_ZCJump)
        STA     R_ChStatus,Y
        CLI
        RTS
;==========================================================================
F_ChPlayTone:
        LDY     R_BackUpCh
L_ChkChBusy:
        LDA     P_SPU_Enable            ;If channel is idle, play note directly; otherwise concatenate jump
        AND     T_ChEnable,Y
        BEQ     L_PlayNoteDirectly
        JMP     L_ConcatenationM

L_PlayNoteDirectly:                     ;When the channel is idle, play note directly
        JSR     F_GetInstAddr           ;Get instrument address

        SEC
        LDY     #01H                    ;Point to base pitch
        LDA     (R_InstAddr),Y          ;Base Pitch - Note Pitch
        SBC     R_MIDITemp              ;A(Base freq) - R_MIDITemp(Note No.)
        BMI     L_BaseIsSmall           ;If overflow, Base is smaller than Note
L_BaseIsLarge:                          ;Note is smaller than Base
        ASL     A
        STA     R_MIDITemp
        SEC
        LDA     #.LOW.(D_PhaseIndexOffset)
        SBC     R_MIDITemp              ;D_PhaseIndexOffset - (Difference*2)
        STA     R_MIDITemp              ;Offset

        INY
        LDA     (R_InstAddr),Y          ;Get index of sampling rate table
        ASL     A
        TAY
        LDA     T_SampleRate,Y
        STA     R_InstTabAddr           ;Get Low address of sampling rate table
        LDA     T_SampleRate+1,Y
        STA     R_InstTabAddr+1         ;Get High address of sampling rate table

        LDY     R_MIDITemp
        LDA     (R_InstTabAddr),Y       ;Get sampling rate LB
        STA     R_MIDITemp              ;Backup sampling rate LB
        INY
        LDA     (R_InstTabAddr),Y       ;Get sampling rate HB
        LDY     R_BackUpCh
        LDX     T_ChMap,Y
        SEI
        STA     R_Ch0FsH,X              ;Save sampling rate HB

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsH,Y
        .ENDIF

        LDA     R_MIDITemp
        STA     R_Ch0FsL,X              ;Save sampling rate LB

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsL,Y
        .ENDIF

        CLI
        JMP     L_SetChPara

L_BaseIsSmall:
        EOR     #FFH                    ;2's complement of Base-Note
        CLC
        ADC     #01H                    ;2's complement
        ASL     A                       ;Each frequency has two bytes
        STA     R_MIDITemp              ;Offset

        INY
        LDA     (R_InstAddr),Y          ;Get index of sampling rate table
        ASL     A                       ;Address is 2 bytes, shift left
        TAY
        LDA     T_SampleRateBase,Y
        STA     R_InstTabAddr           ;Low address
        LDA     T_SampleRateBase+1,Y
        STA     R_InstTabAddr+1         ;High address

        LDY     R_MIDITemp
        LDA     (R_InstTabAddr),Y       ;Get sampling rate LB
        STA     R_MIDITemp              ;Backup sampling rate LB
        INY
        LDA     (R_InstTabAddr),Y       ;Get sampling rate HB
        LDY     R_BackUpCh
        LDX     T_ChMap,Y
        SEI
        STA     R_Ch0FsH,X              ;Save sampling rate HB

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsH,Y
        .ENDIF

        LDA     R_MIDITemp
        STA     R_Ch0FsL,X              ;Save sampling rate LB
 
        .IF PITCH_BEND = ON
        STA     R_ChNoteFsL,Y
        .ENDIF
 
        CLI

L_SetChPara:
        LDA     #00H                    ;Set S1 and S2 as 0 to avoid noise
        STA     R_Ch0S1,X
        STA     R_Ch0S2,X
        STA     R_Ch0PhAddL,X
        STA     R_Ch0PhAddH,X

        LDY     #03H                    ;Point to instrument table address
        LDA     (R_InstAddr),Y
        STA     R_InstTabAddr           ;Save low address of instrument table
        INY
        LDA     (R_InstAddr),Y
        STA     R_InstTabAddr+1         ;Save high address of instrument table

        LDY     #02H
        LDA     (R_InstTabAddr),Y       ;High address of Attack
        STA     R_Ch0DPTRH,X
        INY
        LDA     (R_InstTabAddr),Y       ;Low address of Attack
        STA     R_Ch0DPTRL,X
        INY
        LDA     (R_InstTabAddr),Y       ;Middle address of Attack
        STA     R_Ch0DPTRM,X
L_ChkMode:
        LDY     #0CH
        LDA     (R_InstTabAddr),Y
        LDY     R_BackUpCh
L_ChkMode1:
        CMP     #01H
        BNE     L_ChkMode2
        JMP     L_PCM_PCM_M
L_ChkMode2:
        CMP     #02H
        BNE     L_ChkMode3
        JMP     L_ADPCM_PCM_M
L_ChkMode3:
        CMP     #03H
        BNE     L_ChkMode4
        JMP     L_ADPCM_ADPCM_M
L_ChkMode4:
        CMP     #04H
        BEQ     L_PCM_ADPCM_M
        RTS
;==========================================================================
;Attack(PCM) + Loop(ADPCM)
;==========================================================================
L_PCM_ADPCM_M:
        LDA     R_VolumeInteger,Y
        ORA     #D_Bit7
        SEI
        STA     R_Ch0VolCtl,X           ;Absoulte jump to next wave
        AND     #~D_Bit7
        STA     R_Ch0VolN,X             ;Set next sample's vloume and relative jump mode
        CLI

        LDY     #05H
        LDA     (R_InstTabAddr),Y       ;High address of Loop
        STA     R_Ch0DPTNH,X
        INY
        LDA     (R_InstTabAddr),Y       ;Low address of Loop
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_InstTabAddr),Y       ;Middle address of Loop
        STA     R_Ch0DPTNM,X

        LDY     R_BackUpCh
        SEI
        LDA     R_ChStatus,Y
        ORA     #D_NotePlaying
        AND     #.NOT.(D_ZCJump+D_AdpcmJumpPcm+D_NoteOffFlag+D_DrumPlaying)
        STA     R_ChStatus,Y            ;Clear note off flag

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl           ;Disable channel IRQ

        LDA     P_SPU_Enable
        ORA     T_ChEnable,Y
        STA     P_SPU_Enable            ;Enable SPU Channel
        CLI
        RTS
;==========================================================================
;Attack(ADPCM) + Loop(ADPCM)
;==========================================================================
L_ADPCM_ADPCM_M:
        SEI
        LDA     #00H
        STA     R_Ch0LpLenL,X           ;Initialize loop length before playing ADPCM
        STA     R_Ch0LpLenH,X
        CLI

        LDA     R_VolumeInteger,Y
        AND     #~D_Bit7                ;Set as relative loop mode
        SEI
        STA     R_Ch0VolCtl,X           ;Set channel volume

        LDA     R_ChStatus,Y
        ORA     #D_NotePlaying
        AND     #.NOT.(D_AdpcmJumpPcm+D_ZCJump+D_NoteOffFlag+D_DrumPlaying)
        STA     R_ChStatus,Y            ;Clear note off flag

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl           ;Disbale channel IRQ

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     P_SPU_Enable
        ORA     T_ChEnable,Y
        STA     P_SPU_Enable            ;Enable SPU channel
        CLI
        RTS
;==========================================================================
;Attack(ADPCM) + Loop(PCM)
;==========================================================================
L_ADPCM_PCM_M:
        SEI
        LDA     #00H
        STA     R_Ch0LpLenL,X           ;Initial loop length before playing ADPCM
        STA     R_Ch0LpLenH,X
        CLI

        LDA     R_VolumeInteger,Y
        ORA     #D_Bit7
        SEI
        STA     R_Ch0VolCtl,X           ;Absoulte jump to next wave
        AND     #~D_Bit7
        STA     R_Ch0VolN,X             ;Set next sample's vloume
        CLI

        LDY     #05H
        LDA     (R_InstTabAddr),Y       ;High address of Loop
        STA     R_Ch0DPTNH,X
        INY
        LDA     (R_InstTabAddr),Y       ;Low address of Loop
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_InstTabAddr),Y       ;Middle address of Loop
        STA     R_Ch0DPTNM,X

        LDY     R_BackUpCh
        SEI
        LDA     R_ChStatus,Y
        ORA     #D_NotePlaying+D_AdpcmJumpPcm
        AND     #.NOT.(D_ZCJump+D_NoteOffFlag+D_DrumPlaying)
        STA     R_ChStatus,Y            ;Clear note off flag

        LDA     T_ChEnable,Y
        STA     P_SPU_Status            ;Clear previous IRQ status, avoid IRQ happen immediately after channel IRQ is enabled

        LDA     P_SPU_IntCtrl
        ORA     T_ChEnable,Y
        STA     P_SPU_IntCtrl

        LDA     P_SPU_Enable
        ORA     T_ChEnable,Y
        STA     P_SPU_Enable            ;Enable SPU Channel
        CLI
        RTS
;==========================================================================
;Attack(PCM) + Loop(PCM)
;==========================================================================
L_PCM_PCM_M:
        LDA     R_VolumeInteger,Y
        AND     #~D_Bit7                ;Relative jump mode
        SEI
        STA     R_Ch0VolCtl,X
        STA     R_Ch0VolN,X
        CLI

        LDY     #08H
        LDA     (R_InstTabAddr),Y       ;Get loop length LB
        STA     R_MIDITemp
        INY
        LDA     (R_InstTabAddr),Y       ;Get loop length HB
        SEI
        STA     R_Ch0LpLenH,X
        LDA     R_MIDITemp
        STA     R_Ch0LpLenL,X
        CLI

        LDY     R_BackUpCh
        SEI
        LDA     R_ChStatus,Y
        ORA     #D_NotePlaying
        AND     #.NOT.(D_AdpcmJumpPcm+D_ZCJump+D_NoteOffFlag+D_DrumPlaying)
        STA     R_ChStatus,Y            ;Clear note off flag

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl           ;Disbale channel IRQ

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     P_SPU_Enable
        ORA     T_ChEnable,Y
        STA     P_SPU_Enable            ;Enable SPU Channel
        CLI
        RTS
;==========================================================================
;When a channel is busy, concatenate jump mode
;==========================================================================
L_ConcatenationM:                       ;When a channel is busy
        LDX     T_ChMap,Y
        JSR     F_GetInstAddr

        LDY     #03H
        LDA     (R_InstAddr),Y          ;Get instrument table LB
        STA     R_InstTabAddr
        INY
        LDA     (R_InstAddr),Y          ;Get instrument table HB
        STA     R_InstTabAddr+1

        LDY     #02H
        LDA     (R_InstTabAddr),Y
        STA     R_Ch0DPTNH,X            ;High address of Attack
        INY
        LDA     (R_InstTabAddr),Y
        STA     R_Ch0DPTNL,X            ;Low address of Attack
        INY
        LDA     (R_InstTabAddr),Y
        STA     R_Ch0DPTNM,X            ;Middle address of Attack

        LDY     R_BackUpCh
        LDA     R_VolumeInteger,Y
        AND     #~D_Bit7
        STA     R_Ch0VolN,X             ;Save volume of Attack

        LDY     #0CH
        LDA     (R_InstTabAddr),Y       ;Get COMBINATION
        BEQ     L_Exit?
        CMP     #05H
        BCC     L_AnyMode_CONC
L_Exit?:
        RTS
;==========================================================================
;Attack + Loop
;==========================================================================
L_AnyMode_CONC:
        LDY     R_BackUpCh
        SEI
        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     R_ChStatus,Y
        ORA     #D_ZCJump
        AND     #.NOT.(D_AdpcmJumpPcm+D_DrumPlaying)
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        ORA     T_ChEnable,Y
        STA     P_SPU_IntCtrl
        CLI

        LDA     P_SPU_Conc
        ORA     T_ChEnable,Y
        STA     P_SPU_Conc              ;Concatenate jump to new sample
L_ExitChPlayTone:
        RTS
;==========================================================================
;X = Drum index; R_BackUpCh = Channel No.
;==========================================================================
F_PlayDrum:
        LDY     R_DrumIndex
        LDA     T_DrumInstL,Y
        STA     R_InstTabAddr
        LDA     T_DrumInstH,Y
        STA     R_InstTabAddr+1

        LDY     #00H
        LDA     (R_InstTabAddr),Y       ;Get sampling rate index
        ASL     A
        TAY
        LDA     T_SampleRateBase,Y
        STA     R_InstAddr
        LDA     T_SampleRateBase+1,Y
        STA     R_InstAddr+1

        LDY     R_BackUpCh
        LDX     T_ChMap,Y

        LDA     P_SPU_Enable
        AND     T_ChEnable,Y
        BEQ     L_PlayDrumDirectly
;==========================================================================
; If bit of P_SPU_Conc was not cleared by SPU, that means SPU can't find Z.C. point to connect next wave properly,
; close P_SPU_Enable first then play speech directly to prevent speech be connected without Z.C.
;==========================================================================
        LDA     P_SPU_Conc
        AND     T_ChEnable,Y
        BNE     L_CloseCh?
        JMP     L_ConcatenationD

L_CloseCh?:
        LDA     P_SPU_Enable
        AND     T_ChDisable,Y
        STA     P_SPU_Enable
L_PlayDrumDirectly:
        LDA     #00H
        STA     R_Ch0S1,X
        STA     R_Ch0S2,X               ;Initialize S1 and S2 before playing
        STA     R_Ch0PhAddL,X
        STA     R_Ch0PhAddH,X

        LDY     #00H
        LDA     (R_InstAddr),Y          ;Get sampling rate LB
        STA     R_MIDITemp              ;Backup sampling rate LB
        INY
        LDA     (R_InstAddr),Y          ;Get sampling rate HB
        LDY     R_BackUpCh
        SEI
        STA     R_Ch0FsH,X              ;Save sampling rate HB

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsH,Y
        .ENDIF

        LDA     R_MIDITemp
        STA     R_Ch0FsL,X              ;Save sampling rate LB

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsL,Y
        .ENDIF

        CLI

        LDY     #01H
        LDA     (R_InstTabAddr),Y       ;High address of Attack
        STA     R_Ch0DPTRH,X
        INY
        LDA     (R_InstTabAddr),Y       ;Low address of Attack
        STA     R_Ch0DPTRL,X
        INY
        LDA     (R_InstTabAddr),Y       ;Middle address of Attack
        STA     R_Ch0DPTRM,X

L_ChkModeD:
        LDY     #0DH
        LDA     (R_InstTabAddr),Y       ;Get COMBINATION
        LDY     R_BackUpCh
L_ChkMode1D:
        CMP     #01H
        BNE     L_ChkMode2D
        JMP     L_PCM_PCM_D
L_ChkMode2D:
        CMP     #02H
        BNE     L_ChkMode3D
        JMP     L_ADPCM_PCM_D
L_ChkMode3D:
        CMP     #03H
        BNE     L_ChkMode4D
        JMP     L_ADPCM_ADPCM_D
L_ChkMode4D:
        CMP     #04H
        BEQ     L_PCM_ADPCM_D
        RTS
;==========================================================================
;Attack(PCM) + Loop(ADPCM)
;==========================================================================
L_PCM_ADPCM_D:
        LDA     R_VolumeInteger,Y
        ORA     #D_Bit7                 ;Set CURRENT as absolute jump mode
        STA     R_Ch0VolCtl,X
        AND     #~D_Bit7                ;Set NEXT as relative jump mode
        STA     R_Ch0VolN,X

        LDY     #08H
        LDA     (R_InstTabAddr),Y       ;High address of Loop
        STA     R_Ch0DPTNH,X
        INY
        LDA     (R_InstTabAddr),Y       ;Low address of Loop
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_InstTabAddr),Y       ;Middle address of Loop
        STA     R_Ch0DPTNM,X

        LDY     R_BackUpCh
        SEI
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_NotePlaying+D_AdpcmJumpPcm+D_NoteOffFlag+D_ZCJump)
        ORA     #D_DrumPlaying          ;Add "D_DrumPlaying" flag
        STA     R_ChStatus,Y

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     P_SPU_Enable
        ORA     T_ChEnable,Y
        STA     P_SPU_Enable

        LDA     R_DrumIndex
        STA     R_ChDrumIndex,Y
        CLI
        RTS
;==========================================================================
;Attack(PCM) + Loop(PCM)
;==========================================================================
L_PCM_PCM_D:
        LDA     R_VolumeInteger,Y
        AND     #~D_Bit7                ;Set as relative jump mode
        STA     R_Ch0VolCtl,X
        STA     R_Ch0VolN,X

        LDY     #0BH
        LDA     (R_InstTabAddr),Y       ;Get loop length LB
        STA     R_MIDITemp
        INY
        LDA     (R_InstTabAddr),Y       ;Get loop length HB
        SEI
        STA     R_Ch0LpLenH,X
        LDA     R_MIDITemp
        STA     R_Ch0LpLenL,X
        CLI

        LDY     R_BackUpCh
        SEI
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_NotePlaying+D_AdpcmJumpPcm+D_NoteOffFlag+D_ZCJump)
        ORA     #D_DrumPlaying          ;Add "D_DrumPlaying" flag
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     P_SPU_Enable
        ORA     T_ChEnable,Y
        STA     P_SPU_Enable

        LDA     R_DrumIndex
        STA     R_ChDrumIndex,Y
        CLI
        RTS
;==========================================================================
;Attack(ADPCM) + Loop(PCM)
;==========================================================================
L_ADPCM_PCM_D:
        SEI
        LDA     #00H
        STA     R_Ch0LpLenL,X
        STA     R_Ch0LpLenH,X
        CLI

        LDA     R_VolumeInteger,Y
        ORA     #D_Bit7                 ;Set CURRENT as absolute jump mode
        STA     R_Ch0VolCtl,X
        AND     #~D_Bit7                ;Set NEXT as relative jump mode
        STA     R_Ch0VolN,X

        LDY     #08H
        LDA     (R_InstTabAddr),Y       ;High address of Loop
        STA     R_Ch0DPTNH,X
        INY
        LDA     (R_InstTabAddr),Y       ;Low address of Loop
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_InstTabAddr),Y       ;Middle address of Loop
        STA     R_Ch0DPTNM,X

        LDY     R_BackUpCh
        SEI
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_NotePlaying+D_NoteOffFlag+D_ZCJump)
        ORA     #D_DrumPlaying+D_AdpcmJumpPcm
        STA     R_ChStatus,Y

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     P_SPU_IntCtrl
        ORA     T_ChEnable,Y
        STA     P_SPU_IntCtrl

        LDA     P_SPU_Enable
        ORA     T_ChEnable,Y
        STA     P_SPU_Enable

        LDA     R_DrumIndex
        STA     R_ChDrumIndex,Y
        CLI
        RTS
;==========================================================================
;Attack(ADPCM) + Loop(ADPCM)
;==========================================================================
L_ADPCM_ADPCM_D:
        SEI
        LDA     #00H
        STA     R_Ch0LpLenL,X
        STA     R_Ch0LpLenH,X
        CLI

        LDA     R_VolumeInteger,Y
        AND     #~D_Bit7                ;Set as relative jump mode
        STA     R_Ch0VolCtl,X           ;Set channel volume 

        SEI
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_NotePlaying+D_AdpcmJumpPcm+D_NoteOffFlag+D_ZCJump)
        ORA     #D_DrumPlaying          ;Add "D_DrumPlaying" flag
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     P_SPU_Enable
        ORA     T_ChEnable,Y
        STA     P_SPU_Enable

        LDA     R_DrumIndex
        STA     R_ChDrumIndex,Y
        CLI
        RTS
;==========================================================================
;When a channel is busy, concatenate jump mode
;==========================================================================
L_ConcatenationD:
        LDY     #01H
        LDA     (R_InstTabAddr),Y       ;High address of Attack
        STA     R_Ch0DPTNH,X
        INY
        LDA     (R_InstTabAddr),Y       ;Low address of Attack
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_InstTabAddr),Y       ;Middle address of Attack
        STA     R_Ch0DPTNM,X

        LDY     R_BackUpCh
        LDA     R_VolumeInteger,Y
        AND     #~D_Bit7                ;Set NEXT as relative jump mode
        STA     R_Ch0VolN,X

        LDY     #0DH
        LDA     (R_InstTabAddr),Y       ;Get COMBINATION
        CMP     #05H
        BCC     L_AnyMode_D_CONC
        RTS
;==========================================================================
;Attack + Loop
;==========================================================================
L_AnyMode_D_CONC:
        LDY     R_BackUpCh
        SEI
        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     R_ChStatus,Y
        AND     #.NOT.(D_NotePlaying+D_AdpcmJumpPcm+D_NoteOffFlag)
        ORA     #D_DrumPlaying+D_ZCJump
        STA     R_ChStatus,Y

        LDA     P_SPU_Conc
        ORA     T_ChEnable,Y
        STA     P_SPU_Conc

        LDA     P_SPU_IntCtrl
        ORA     T_ChEnable,Y
        STA     P_SPU_IntCtrl

        LDA     R_DrumIndex
        STA     R_ChDrumIndex,Y
        CLI
L_ExitPlayDrum:
        RTS
;==========================================================================
F_CheckADSR:
        LDA     R_MIDIStatus
        AND     #D_MusicOff
        BEQ     L_PlayM?
        JMP     L_ExitChkChADSR

L_PlayM?:
        LDA     R_MIDIStatus
        BMI     L_ProcessChADSR         ;Bit7 = 1, MIDI is played
        LDA     R_SingleNote            ;Check if a single note or drum is played
        ORA     R_SingleDrum
        BNE     L_ProcessChADSR
        JMP     L_ExitChkChADSR

L_ProcessChADSR:
        LDY     #00H
        STY     R_ADSRChannel           ;R_ADSRChannel = Channel number
L_ChkChADSR:
        LDY     R_ADSRChannel
        LDX     T_ChMap,Y

        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,Y
        AND     #D_SpeechPlaying
        BNE     L_ChkNextChADSR
        .ENDIF

        LDA     R_EnvTimeBase,Y
        BNE     L_ChkNextChADSR         ;Process envelope when TimeBase = 0

        LDA     R_ChStatus,Y
        AND     #D_NoteOffFlag
        BNE     L_NoteOffRelease        ;If Mididuration = 0, note release

        LDA     R_ChStatus,Y
        AND     #D_ReachLastDrop
        BNE     L_ChkNextChADSR         ;If = 1, the envelope is reach last drop, skip envelope module 

        LDA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        BEQ     L_ChkNextChADSR

;==========================================================================
;Avoid updating envelope before concatenation
;==========================================================================
        LDA     P_SPU_Conc
        AND     T_ChEnable,Y
        BNE     L_ChkNextChADSR
        JSR     F_CheckChADSR
        JMP     L_ChkNextChADSR
L_NoteOffRelease:
        LDA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        BEQ     L_ChkNextChADSR
        JSR     F_NoteOff
L_ChkNextChADSR:
        CLI
        INC     R_ADSRChannel
        LDY     R_ADSRChannel
        CPY     #D_ChannelNo
        BEQ     L_ExitChkChADSR
        JMP     L_ChkChADSR
L_ExitChkChADSR:
        RTS
;==========================================================================
;Target and slope both need to multiply the value of "R_Velovol"
;==========================================================================
F_CheckChADSR:
        SEI
        LDA     R_ADSRTimeBase
        STA     R_EnvTimeBase,Y         ;Reset counter
        CLI
        LDX     T_ChMap,Y
        LDA     R_SlopeInteger,Y
        BPL     L_AddEnvelope
        JMP     L_SubEnvelope

L_AddEnvelope:
        CLC
        LDA     R_VolumeFloat,Y         ;(Float of Volume) + (Float of Slope)
        ADC     R_SlopeFloat,Y
        STA     R_VolumeFloat,Y

        LDA     R_VolumeInteger,Y
        AND     #~D_Bit7
        ADC     R_SlopeInteger,Y        ;(Integer of Volume) + (Integer of Slope)
        BMI     L_IsTarget_
        STA     R_VolumeInteger,Y

        SEC
        LDA     R_TargetFloat,Y
        SBC     R_VolumeFloat,Y
        LDA     R_TargetInteger,Y
        SBC     R_VolumeInteger,Y
        BCC     L_IsTarget_             ;If Target < Current, the envelope reached target
L_IsNotTarget_:
        SEI
        LDA     R_Ch0VolCtl,X
        AND     #D_Bit7
        ORA     R_VolumeInteger,Y
        STA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X
        CLI
        JMP     L_ExitCheckChADSR
L_IsTarget_:                            ;Reach target
        SEI
        LDA     R_Ch0VolCtl,X
        AND     #D_Bit7
        ORA     R_TargetInteger,Y
        STA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X
        LDA     R_ChStatus,Y
        AND     #D_TargetIsLastDrop
        BEQ     L_NotReachLastDrop_?
L_ReachLastDrop_?:
        LDA     R_ChStatus,Y
        ORA     #D_ReachLastDrop
        STA     R_ChStatus,Y
        CLI
        JMP     L_ExitCheckChADSR
L_NotReachLastDrop_?:
        CLI
        LDA     R_TargetInteger,Y
        STA     R_VolumeInteger,Y       ;Volume integer for next time

        LDA     R_TargetFloat,Y
        STA     R_VolumeFloat,Y

        CLC
        LDA     R_TabOffset,Y
        ADC     #04H
        STA     R_ADSROffset
        STA     R_TabOffset,Y           ;Get next ADSR value

        LDA     R_ChStatus,Y
        AND     #D_DrumPlaying
        BNE     L_GetDrumAdsrValue_
        JSR     F_GetAdsrValue          ;Input = R_ADSROffset; Output = Acc.
        JMP     L_GetSlope_
L_GetDrumAdsrValue_:
        LDA     R_ChDrumIndex,Y
        TAY
        JSR     F_GetDrumAdsrValue
L_GetSlope_:
        LDY     #01H
        LDA     (R_InstAddr),Y          ;Get integer of slope
        BMI     L_SlopeMinus_           ;Check if slope is minus
L_SlopePlus_:                           ;Slope is plus
        STA     R_MultiplierM           ;Save integer of slope
        INY
        LDA     (R_InstAddr),Y          ;Get float of slope
        STA     R_MultiplierL           ;Save float of slope
        LDY     R_ADSRChannel
        LDA     R_VeloVol,Y
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        ORA     R_ProductM
        BNE     L_Branch?
        LDA     #01H                    ;If slop is zero, set "01H" as minimum value
        STA     R_ProductM
L_Branch?:
        LDA     R_ProductH
        STA     R_SlopeInteger,Y
        LDA     R_ProductM
        STA     R_SlopeFloat,Y
        JMP     L_GetTarget_

L_SlopeMinus_:                          ;Slope is minus
        EOR     #FFH
        STA     R_MultiplierM           ;Save integer of slope
        INY
        LDA     (R_InstAddr),Y          ;Get float of slope
        EOR     #FFH
        CLC
        ADC     #01H
        STA     R_MultiplierL           ;Save float of slope
        LDY     R_ADSRChannel
        LDA     R_VeloVol,Y
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        ORA     R_ProductM
        BNE     L_Branch?
        LDA     #01H                    ;If slop is zero, set "01H" as minimum value
        STA     R_ProductM
L_Branch?:
        LDA     R_ProductH
        EOR     #FFH
        STA     R_SlopeInteger,Y
        CLC
        LDA     R_ProductM
        EOR     #FFH
        ADC     #01H
        STA     R_SlopeFloat,Y

L_GetTarget_:
        LDY     #03H
        LDA     (R_InstAddr),Y          ;Get target
        ASL     A
        STA     P_MulF
        LDY     R_ADSRChannel
        LDA     R_VeloVol,Y
        STA     P_MulM
        LDA     #D_MulAct
        STA     P_MulAct
L_Wait?:
        BIT     P_MulAct
        BNE     L_Wait?

        LDA     P_MulOutH
        STA     R_TargetInteger,Y       ;Get next integer part of Target
        LDA     P_MulOutL
        STA     R_TargetFloat,Y         ;Get next float part of Target

        LDY     #07H
        LDA     (R_InstAddr),Y          ;Get next target
        CMP     #FFH
        BEQ     L_TargetIsLastDrop_?
        JMP     L_ExitCheckChADSR

L_TargetIsLastDrop_?:
        LDY     R_ADSRChannel
        SEI
        LDA     R_ChStatus,Y
        ORA     #D_TargetIsLastDrop
        STA     R_ChStatus,Y
        CLI
        JMP     L_ExitCheckChADSR

L_SubEnvelope:
        CLC
        LDA     R_VolumeFloat,Y         ;(Float of Volume) - (Float of Slope)
        ADC     R_SlopeFloat,Y
        STA     R_VolumeFloat,Y

        LDA     R_VolumeInteger,Y
        AND     #~D_Bit7
        ADC     R_SlopeInteger,Y        ;(Integer of Volume) - (Integer of Slope)
        BMI     L_IsTarget
        STA     R_VolumeInteger,Y

        SEC
        LDA     R_VolumeFloat,Y
        SBC     R_TargetFloat,Y
        LDA     R_VolumeInteger,Y
        SBC     R_TargetInteger,Y
        BCC     L_IsTarget              ;If Target > Current, the envelope reached target
L_IsNotTarget:
        SEI
        LDA     R_Ch0VolCtl,X
        AND     #D_Bit7
        ORA     R_VolumeInteger,Y
        STA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X
        CLI
        JMP     L_ExitCheckChADSR
L_IsTarget:                             ;Reach target
        SEI
        LDA     R_Ch0VolCtl,X
        AND     #D_Bit7
        ORA     R_TargetInteger,Y
        STA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X
        CLI
        BNE     L_EnvelopeLastDropIsNotZero
L_EnvelopeLastDropIsZero:
        LDA     #00H
        STA     R_TabOffset,Y           ;Clear table offset
        STA     R_VolumeFloat,Y         ;Reset float part of volume to 0

        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,Y
        AND     #D_SpeechPlaying
        BNE     L_ExitCheckChADSR
        .ENDIF

        JMP     F_StopSound             ;Stop channel
L_ExitCheckChADSR:
        RTS

L_EnvelopeLastDropIsNotZero:
        LDA     R_ChStatus,Y
        AND     #D_TargetIsLastDrop
        BEQ     L_NotReachLastDrop?     ;Check the current target is last drop or not
L_ReachLastDrop?:
        SEI
        LDA     R_ChStatus,Y
        ORA     #D_ReachLastDrop
        STA     R_ChStatus,Y            ;Reach last drop, set flag to skip envelope module
        CLI
        JMP     L_ExitCheckChADSR
L_NotReachLastDrop?:
        LDA     R_TargetInteger,Y
        STA     R_VolumeInteger,Y

        LDA     R_TargetFloat,Y
        STA     R_VolumeFloat,Y

        CLC
        LDA     R_TabOffset,Y
        ADC     #04H
        STA     R_ADSROffset
        STA     R_TabOffset,Y           ;Get next ADSR value

        LDA     R_ChStatus,Y
        AND     #D_DrumPlaying
        BNE     L_GetDrumAdsrValue
        JSR     F_GetAdsrValue          ;Input = R_ADSROffset; Output = Acc.
        JMP     L_GetSlope
L_GetDrumAdsrValue:
        LDA     R_ChDrumIndex,Y
        TAY
        JSR     F_GetDrumAdsrValue
L_GetSlope:
        LDY     #01H
        LDA     (R_InstAddr),Y          ;Get integer of slope
        BMI     L_SlopeMinus
L_SlopePlus:
        STA     R_MultiplierM           ;Save integer of slope
        INY
        LDA     (R_InstAddr),Y          ;Get float of slope
        STA     R_MultiplierL           ;Save float of slope
        LDY     R_ADSRChannel
        LDA     R_VeloVol,Y             ;Volume X Velocity
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        ORA     R_ProductM
        BNE     L_Branch?
        LDA     #01H                    ;If slop is zero, set "01H" as minimum value
        STA     R_ProductM
L_Branch?:
        LDA     R_ProductH
        STA     R_SlopeInteger,Y
        LDA     R_ProductM
        STA     R_SlopeFloat,Y
        JMP     L_GetTarget

L_SlopeMinus:
        EOR     #FFH
        STA     R_MultiplierM           ;Save integer of slope
        INY
        LDA     (R_InstAddr),Y          ;Get float of slope
        EOR     #FFH
        CLC
        ADC     #01H
        STA     R_MultiplierL           ;Save float of slope
        LDY     R_ADSRChannel
        LDA     R_VeloVol,Y
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        ORA     R_ProductM
        BNE     L_Branch?
        LDA     #01H                    ;If slop is zero, set "01H" as minimum value
        STA     R_ProductM
L_Branch?:
        LDA     R_ProductH
        EOR     #FFH
        STA     R_SlopeInteger,Y
        LDA     R_ProductM
        EOR     #FFH
        CLC
        ADC     #01H
        STA     R_SlopeFloat,Y
L_GetTarget:
        LDY     #03H
        LDA     (R_InstAddr),Y          ;Get target
        ASL     A
        STA     P_MulF
        LDY     R_ADSRChannel
        LDA     R_VeloVol,Y
        STA     P_MulM
        LDA     #D_MulAct
        STA     P_MulAct
L_Wait?:
        BIT     P_MulAct
        BNE     L_Wait?

        LDA     P_MulOutH
        STA     R_TargetInteger,Y
        LDA     P_MulOutL
        STA     R_TargetFloat,Y

        LDY     #07H
        LDA     (R_InstAddr),Y          ;Get next target
        CMP     #FFH
        BEQ     L_TargetIsLastDrop?
        JMP     L_ExitCheckChADSR       ;Target is not last drop
L_TargetIsLastDrop?:
        LDY     R_ADSRChannel
        SEI
        LDA     R_ChStatus,Y
        ORA     #D_TargetIsLastDrop
        STA     R_ChStatus,Y            ;Set flag to indicate the current target is last drop in the envelope table
        CLI
        JMP     L_ExitCheckChADSR
;==========================================================================
;Input = R_ADSRChannel = Channel number (0~7)
;==========================================================================
F_NoteOff:
        LDY     R_ADSRChannel
        LDA     R_ChStatus,Y            ;Initialize note off envelope is just done once, set a flag to skip F_InitialNoteOffEnv
        AND     #D_SkipIniNoteOffEnv
        BNE     L_SkipIniNoteOffEnv?

        JSR     F_InitialNoteOffEnv

        SEI
        LDY     R_ADSRChannel
        LDA     R_ChStatus,Y
        ORA     #D_SkipIniNoteOffEnv
        STA     R_ChStatus,Y
        CLI
L_SkipIniNoteOffEnv?:
        LDX     T_ChMap,Y
        SEI
        LDA     R_NoteOffTimeBase
        STA     R_EnvTimeBase,Y         ;Reset envelope time base
        CLI

        CLC
        LDA     R_VolumeFloat,Y
        ADC     R_SlopeFloat,Y
        STA     R_VolumeFloat,Y
        LDA     R_Ch0VolCtl,X           ;Current volume
        AND     #~D_Bit7
        ADC     R_SlopeInteger,Y
        BMI     L_IsTargetNoteOff       ;If minus, the envelope already reach zero
        STA     R_VolumeInteger,Y

        SEC
        LDA     R_VolumeFloat,Y
        SBC     R_TargetFloat,Y
        LDA     R_VolumeInteger,Y
        SBC     R_TargetInteger,Y
        BCC     L_IsTargetNoteOff       ;If Target > Current, note off is finished
L_IsNotTargetNoteOff:
        SEI
        LDA     R_Ch0VolCtl,X
        AND     #D_Bit7
        ORA     R_VolumeInteger,Y
        STA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X
        CLI
        BNE     L_ExitNoteOff           ;Close Ch when Volume = 0
        JMP     F_StopSound
L_ExitNoteOff:
        RTS

L_IsTargetNoteOff:
        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,Y
        AND     #D_SpeechPlaying
        BNE     L_ExitNoteOff
        .ENDIF

        SEI
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_NoteOffFlag+D_TargetIsLastDrop+D_ReachLastDrop+D_ZCJump+D_SkipIniNoteOffEnv+D_NotePlaying+D_DrumPlaying+D_AdpcmJumpPcm)
        STA     R_ChStatus,Y
        CLI
;==========================================================================
F_StopSound:                            ;Y = Channel number
        LDX     T_ChMap,Y
        SEI
        LDA     R_Ch0VolCtl,X
        AND     #D_Bit7
        STA     R_Ch0VolCtl,X           ;Close channel volume

        LDA     P_SPU_Enable
        AND     T_ChDisable,Y
        STA     P_SPU_Enable            ;Close the channel

        LDA     #00H
        STA     R_ChTime,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl           ;Close the channel's IRQ

        LDA     R_SingleNote
        AND     T_ChDisable,Y
        STA     R_SingleNote            ;If a single note is played, clear flag

        LDA     R_SingleDrum
        AND     T_ChDisable,Y
        STA     R_SingleDrum            ;If a single drum is played, clear flag

        CLI
L_ExitStopSound:
        RTS
;==========================================================================
F_InitialNoteOffEnv:
        JSR     F_GetNoteOffEnv         ;Get address of note off table
        LDY     #00H
        LDA     (R_InstAddr),Y          ;Get integer of slope
        BMI     L_SlopeMinusNoteOff     ;Check if the slope is minus
L_SlopePlusNoteOff:
        STA     R_MultiplierM
        INY
        LDA     (R_InstAddr),Y          ;Get float of slope
        STA     R_MultiplierL
        LDY     R_ADSRChannel
        LDA     R_VeloVol,Y
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        STA     R_SlopeInteger,Y
        LDA     R_ProductM
        STA     R_SlopeFloat,Y
        JMP     L_GetTargetNoteOff

L_SlopeMinusNoteOff:
        EOR     #FFH
        STA     R_MultiplierM
        INY
        LDA     (R_InstAddr),Y          ;Get float of slope
        EOR     #FFH
        CLC
        ADC     #01H
        STA     R_MultiplierL
        LDY     R_ADSRChannel
        LDA     R_VeloVol,Y
        ASL     A
        STA     R_FaciendL
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        EOR     #FFH
        STA     R_SlopeInteger,Y        ;Get integer of slope which is multiplied by Volume X Velocity
        LDA     R_ProductM
        EOR     #FFH
        CLC
        ADC     #01H
        STA     R_SlopeFloat,Y          ;Get float of slope which is multiplied by Volume X Velocity
L_GetTargetNoteOff:
        LDY     #02H
        LDA     (R_InstAddr),Y          ;Get target
        ASL     A
        STA     P_MulF
        LDY     R_ADSRChannel
        LDA     R_VeloVol,Y
        STA     P_MulM
        LDA     #D_MulAct
        STA     P_MulAct
L_Wait?:
        BIT     P_MulAct
        BNE     L_Wait?

        LDA     P_MulOutH
        STA     R_TargetInteger,Y
        LDA     P_MulOutL
        STA     R_TargetFloat,Y
L_ExitInitialNoteOffADSR:
        RTS
;==========================================================================
F_GetNoteOffEnv:                        ;Get address of note off table
        LDA     R_ChStatus,Y
        AND     #D_DrumPlaying
        BEQ     L_GetInstNoteOffEnv
L_GetDrumNoteOffEnv:
        LDA     R_ChDrumIndex,Y         ;Get drum index
        TAY
        LDA     T_DrumInstL,Y
        STA     R_InstTabAddr           ;Get low address of drum table
        LDA     T_DrumInstH,Y
        STA     R_InstTabAddr+1         ;Get high address of drum table

        LDY     #06H
        LDA     (R_InstTabAddr),Y       ;Get low address of note off table
        STA     R_InstAddr
        INY
        LDA     (R_InstTabAddr),Y       ;Get high address of note off table
        STA     R_InstAddr+1
        RTS

L_GetInstNoteOffEnv:
        JSR     F_GetInstAddr           ;Get instrument address

        LDY     #03H
        LDA     (R_InstAddr),Y          ;Get low address of instrument table
        STA     R_InstTabAddr
        INY
        LDA     (R_InstAddr),Y          ;Get high address of instrument table
        STA     R_InstTabAddr+1

        LDY     #0AH
        LDA     (R_InstTabAddr),Y       ;Get low address of note off table
        STA     R_InstAddr
        INY
        LDA     (R_InstTabAddr),Y       ;Get high address of note off table
        STA     R_InstAddr+1
        RTS
;==========================================================================
F_GetAdsrValue:                         ;Input = R_ADSROffset; Output = Acc
        JSR     F_GetInstAddr           ;R_InstAddr : Low address, R_InstAddr+1 : High address

        LDY     #03H
        LDA     (R_InstAddr),Y
        STA     R_InstTabAddr           ;Low address of instrument table
        INY
        LDA     (R_InstAddr),Y
        STA     R_InstTabAddr+1         ;High address of instrument table

        LDY     #00H
        LDA     (R_InstTabAddr),Y
        STA     R_InstAddr              ;Low address of envelope table
        INY
        LDA     (R_InstTabAddr),Y
        STA     R_InstAddr+1            ;High address of envelope table

        %M_IncreaseAddr R_InstAddr,R_ADSROffset 
        LDY     #00H
        LDA     (R_InstAddr),Y          ;Get first data of envelope
        RTS
;==========================================================================
F_GetInstAddr:                          ;R_InstAddr = Low address, R_InstAddr+1 = High address 
        LDA     R_Note,Y                ;Note pitch
        STA     R_MIDITemp
        LDA     R_InstIndex,Y           ;Get the MIDI channel of current SPU channel
        TAY
        LDA     R_ChInst,Y              ;Get instrument index for this MIDI channel
        TAY
        LDA     T_InstrumentsL,Y
        STA     R_InstAddr              ;Instrument address LB
        LDA     T_InstrumentsH,Y
        STA     R_InstAddr+1            ;Instrument address HB
        LDY     #00H
L_GetMaxPitch:
        LDA     (R_InstAddr),Y          ;Get Max Pitch of searched instrument
        BEQ     L_IsEnd                 ;If instrument table content is 0, exit
        CMP     R_MIDITemp              ;Compare with note pitch
        BCS     L_IsEnd                 ;If max pitch >= note pitch, get it
        CLC
        LDA     R_InstAddr              ;Address+5
        ADC     #05H                    ;Get next max pitch
        STA     R_InstAddr
        BCC     L_GetMaxPitch
        INC     R_InstAddr+1
        JMP     L_GetMaxPitch
L_IsEnd:
        RTS
;==========================================================================
F_GetDrumAdsrValue:
        LDA     T_DrumInstL,Y
        STA     R_InstTabAddr
        LDA     T_DrumInstH,Y
        STA     R_InstTabAddr+1

        LDY     #04H
        LDA     (R_InstTabAddr),Y
        STA     R_InstAddr              ;Low address of envelope table
        INY
        LDA     (R_InstTabAddr),Y
        STA     R_InstAddr+1            ;High address of envelope table

        %M_IncreaseAddr R_InstAddr,R_ADSROffset
        LDY     #00H
        LDA     (R_InstAddr),Y          ;Get first data of envelope
        RTS
;==========================================================================
;Set the parameters of next sample (put in V_IRQ_SPU)
;==========================================================================
F_SpuIrqMIDIService:
        LDA     R_MIDIStatus
        AND     #D_MIDIEn
        BNE     L_GO?
        LDA     R_SingleNote            ;Check if a single note or drum is played
        ORA     R_SingleDrum
        BNE     L_GO?
        RTS
L_GO?:
        LDA     P_SPU_Status
        AND     P_SPU_IntCtrl
        STA     R_ConcChannel           ;Backup channels which issued IRQ
        LDA     R_ConcChannel
L_Precede:                              ;Process all interrupt event
        LDY     #00H
        ROR     R_ConcChannel
        BCC     L_NoBit0?
        JSR     F_SetMIDIIrqPara
L_NoBit0?:
        INY
        ROR     R_ConcChannel
        BCC     L_NoBit1?
        JSR     F_SetMIDIIrqPara
L_NoBit1?:
        INY
        ROR     R_ConcChannel
        BCC     L_NoBit2?
        JSR     F_SetMIDIIrqPara
L_NoBit2?:
        INY
        ROR     R_ConcChannel
        BCC     L_NoBit3?
        JSR     F_SetMIDIIrqPara
L_NoBit3?:
        INY
        ROR     R_ConcChannel
        BCC     L_NoBit4?
        JSR     F_SetMIDIIrqPara
L_NoBit4?:
        INY
        ROR     R_ConcChannel
        BCC     L_NoBit5?
        JSR     F_SetMIDIIrqPara
L_NoBit5?:
        INY
        ROR     R_ConcChannel
        BCC     L_NoBit6?
        JSR     F_SetMIDIIrqPara
L_NoBit6?:
        INY
        ROR     R_ConcChannel
        BCC     L_NoBit7?
        JSR     F_SetMIDIIrqPara
L_NoBit7?:
        RTS
;==========================================================================
F_SetMIDIIrqPara:
        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,Y
        AND     #D_SpeechPlaying
        BEQ     L_GoChkMidiIrqFlag
        RTS
        .ENDIF

L_GoChkMidiIrqFlag:
        STY     R_BackUpChIRQ           ;Backup channel
        LDX     T_ChMap,Y
        LDA     R_ChStatus,Y
        AND     #D_DrumPlaying          ;Check if drum is played
        BEQ     L_GoOn?
        JMP     L_GoChkDrumIrqFlag
;==========================================================================
;Play instrument
;==========================================================================
L_GoOn?:
        LDA     R_ChStatus,Y
        AND     #D_ZCJump               ;Check concatenate jump mode
        BEQ     L_ChkAJP?
        JMP     L_InstZCJumpIRQ

L_ChkAJP?:
        LDA     R_ChStatus,Y
        AND     #D_AdpcmJumpPcm         ;Check ADPCM jump to PCM
        BNE     L_AdpcmJumpPcmIRQ

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl           ;Disable channel IRQ

        LDA     T_ChEnable,Y
        STA     P_SPU_Status            ;Clear INT status
        RTS
;==========================================================================
;ADPCM jump to PCM
;==========================================================================
L_AdpcmJumpPcmIRQ:
        LDA     R_Note,Y                ;Note pitch
        STA     R_MIDITempIRQ
        LDA     R_InstIndex,Y           ;Get the MIDI channel of current SPU channel
        TAY
        LDA     R_ChInst,Y              ;Get instrument index
        TAY
        LDA     T_InstrumentsL,Y
        STA     R_InstAddrIRQ           ;Instrument address LB
        LDA     T_InstrumentsH,Y
        STA     R_InstAddrIRQ+1         ;Instrument address HB
        LDY     #00H
L_GetMaxPitch?:
        LDA     (R_InstAddrIRQ),Y       ;Get max pitch
        BEQ     L_IsEndIRQ?             ;If instrument table content is 0, exit
        CMP     R_MIDITempIRQ           ;Compare with current note
        BCS     L_IsEndIRQ?             ;If max pitch >= note pitch, get it
        CLC
        LDA     R_InstAddrIRQ           ;Address+5
        ADC     #05H                    ;Get next max pitch
        STA     R_InstAddrIRQ
        BCC     L_GetMaxPitch?
        INC     R_InstAddrIRQ+1
        JMP     L_GetMaxPitch?
L_IsEndIRQ?:
        LDY     #03H
        LDA     (R_InstAddrIRQ),Y       ;Get instrument table address LB
        STA     R_InstTabAddrIRQ
        INY
        LDA     (R_InstAddrIRQ),Y       ;Get instrument table address HB
        STA     R_InstTabAddrIRQ+1

        LDY     #08H
        LDA     (R_InstTabAddrIRQ),Y    ;Get loop length LB
        STA     R_Ch0LpLenL,X
        INY
        LDA     (R_InstTabAddrIRQ),Y    ;Get loop length HB
        STA     R_Ch0LpLenH,X

        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_AdpcmJumpPcm)
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl           ;Disable channel IRQ

        LDA     T_ChEnable,Y
        STA     P_SPU_Status
        RTS
;==========================================================================
;Concatenate jump mode
;==========================================================================
L_InstZCJumpIRQ:
        LDA     R_Note,Y                ;Note pitch
        STA     R_MIDITempIRQ
        LDA     R_InstIndex,Y           ;Get the MIDI channel of current SPU channel
        TAY
        LDA     R_ChInst,Y              ;Get instrument index
        TAY
        LDA     T_InstrumentsL,Y
        STA     R_InstAddrIRQ           ;Instrument address LB
        LDA     T_InstrumentsH,Y
        STA     R_InstAddrIRQ+1         ;Instrument address HB
        LDY     #00H
L_GetMaxPitch?:
        LDA     (R_InstAddrIRQ),Y       ;Get max pitch
        BEQ     L_IsEndIRQ?             ;If instrument table content is 0, exit
        CMP     R_MIDITempIRQ           ;Compare with current note
        BCS     L_IsEndIRQ?             ;If max pitch >= note pitch, get it
        CLC
        LDA     R_InstAddrIRQ           ;Address+5
        ADC     #05H                    ;Get next max pitch
        STA     R_InstAddrIRQ
        BCC     L_GetMaxPitch?
        INC     R_InstAddrIRQ+1
        JMP     L_GetMaxPitch?
L_IsEndIRQ?:
        SEC
        LDY     #01H
        LDA     (R_InstAddrIRQ),Y       ;Base Pitch - Note Pitch
        SBC     R_MIDITempIRQ
        BMI     L_BaseIsSmallIRQ        ;If overflow, Base is smaller than Note
L_BaseIsLargeIRQ:                       ;Note pitch is smaller than Base
        ASL     A
        STA     R_MIDITempIRQ
        SEC
        LDA     #.LOW.(D_PhaseIndexOffset)
        SBC     R_MIDITempIRQ           ;D_PhaseIndexOffset - (Difference*2)
        STA     R_MIDITempIRQ           ;Offset

        LDY     #02H
        LDA     (R_InstAddrIRQ),Y       ;Table index of sampling rate
        ASL     A
        TAY
        LDA     T_SampleRate,Y
        STA     R_InstTabAddrIRQ        ;Get low address of sampling rate table
        LDA     T_SampleRate+1,Y
        STA     R_InstTabAddrIRQ+1      ;Get high address of sampling rate table

        LDY     R_MIDITempIRQ
        LDA     (R_InstTabAddrIRQ),Y    ;Get sampling rate LB
        STA     R_MIDITempIRQ
        INY
        LDA     (R_InstTabAddrIRQ),Y    ;Get sampling rate HB
        LDY     R_BackUpChIRQ
        STA     R_Ch0FsH,X

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsH,Y
        .ENDIF

        LDA     R_MIDITempIRQ
        STA     R_Ch0FsL,X

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsL,Y
        .ENDIF

        JMP     L_SetChParaIRQ

L_BaseIsSmallIRQ:
        EOR     #FFH                    ;2's complement of Base-Note
        CLC
        ADC     #01H                    ;2's complement
        ASL     A                       ;Every frequency has 2 bytes
        STA     R_MIDITempIRQ           ;Offset

        LDY     #02H
        LDA     (R_InstAddrIRQ),Y       ;Table index of sampling rate
        ASL     A
        TAY
        LDA     T_SampleRateBase,Y
        STA     R_InstTabAddrIRQ        ;Get low address of sampling rate table
        LDA     T_SampleRateBase+1,Y
        STA     R_InstTabAddrIRQ+1      ;Get high address of sampling rate table

        LDY     R_MIDITempIRQ
        LDA     (R_InstTabAddrIRQ),Y    ;Get sampling rate LB
        STA     R_MIDITempIRQ
        INY
        LDA     (R_InstTabAddrIRQ),Y    ;Get sampling rate HB
        LDY     R_BackUpChIRQ
        STA     R_Ch0FsH,X

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsH,Y
        .ENDIF

        LDA     R_MIDITempIRQ
        STA     R_Ch0FsL,X

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsL,Y
        .ENDIF

L_SetChParaIRQ:
        LDY     #03H
        LDA     (R_InstAddrIRQ),Y       ;Get instrument table address LB
        STA     R_InstTabAddrIRQ
        INY
        LDA     (R_InstAddrIRQ),Y       ;Get instrument table address HB
        STA     R_InstTabAddrIRQ+1
L_ChkInstModeInZCIrq:
        LDY     #02H
        LDA     (R_InstTabAddrIRQ),Y    ;Get high address of CURRENT
        BMI     L_AttackIsPCM
L_AttackIsADPCM:
        LDY     #05H
        LDA     (R_InstTabAddrIRQ),Y    ;Get high address of NEXT
        BPL     L_BothADPCM_ZCIrq
        JMP     L_ADPCM_PCM_ZCIrq
L_AttackIsPCM:
        LDY     #05H
        LDA     (R_InstTabAddrIRQ),Y
        BMI     L_BothPCM_ZCIrq
;==========================================================================
;Attack(PCM) + Loop(ADPCM)
;==========================================================================
L_PCM_ADPCM_ZCIrq:
        LDA     (R_InstTabAddrIRQ),Y    ;Set address of NEXT
        STA     R_Ch0DPTNH,X
        INY
        LDA     (R_InstTabAddrIRQ),Y
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_InstTabAddrIRQ),Y
        STA     R_Ch0DPTNM,X

        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        ORA     #D_NotePlaying
        AND     #.NOT.(D_ZCJump+D_AdpcmJumpPcm+D_NoteOffFlag+D_DrumPlaying)
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     R_Ch0VolCtl,X
        ORA     #D_Bit7
        STA     R_Ch0VolCtl,X           ;Set as absolute jump mode

        LDA     R_Ch0VolN,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X             ;Set as relative jump mode
        RTS
;==========================================================================
;Attack(ADPCM) + Loop(ADPCM)
;==========================================================================
L_BothADPCM_ZCIrq:
        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        ORA     #D_NotePlaying
        AND     #.NOT.(D_ZCJump+D_AdpcmJumpPcm+D_NoteOffFlag+D_DrumPlaying)
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        STA     R_Ch0VolCtl,X           ;Set as relative jump mode

        LDA     R_Ch0VolN,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X             ;Set as relative jump mode
        RTS
;==========================================================================
;Attack(PCM) + Loop(PCM)
;==========================================================================
L_BothPCM_ZCIrq:
        LDY     #08H
        LDA     (R_InstTabAddrIRQ),Y    ;Get loop length LB
        STA     R_Ch0LpLenL,X
        INY
        LDA     (R_InstTabAddrIRQ),Y    ;Get loop length HB
        STA     R_Ch0LpLenH,X

        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        ORA     #D_NotePlaying
        AND     #.NOT.(D_ZCJump+D_AdpcmJumpPcm+D_NoteOffFlag+D_DrumPlaying)
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        STA     R_Ch0VolCtl,X           ;Set as relative jump mode

        LDA     R_Ch0VolN,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X             ;Set as relative jump mode
        RTS
;==========================================================================
;Attack(ADPCM) + Loop(PCM)
;==========================================================================
L_ADPCM_PCM_ZCIrq:
        LDA     (R_InstTabAddrIRQ),Y    ;Set address of NEXT
        STA     R_Ch0DPTNH,X
        INY
        LDA     (R_InstTabAddrIRQ),Y
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_InstTabAddrIRQ),Y
        STA     R_Ch0DPTNM,X

        LDA     R_Ch0VolCtl,X
        ORA     #D_Bit7                 ;Set as absolute jump mode
        STA     R_Ch0VolCtl,X

        LDA     R_Ch0VolN,X
        AND     #~D_Bit7                ;Set as relative jump mode
        STA     R_Ch0VolN,X

        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        ORA     #D_AdpcmJumpPcm+D_NotePlaying
        AND     #.NOT.(D_ZCJump+D_NoteOffFlag+D_DrumPlaying)
        STA     R_ChStatus,Y

        LDA     T_ChEnable,Y
        STA     P_SPU_Status
        RTS
;==========================================================================
;Play drum
;==========================================================================
L_GoChkDrumIrqFlag:
        LDA     R_ChStatus,Y
        AND     #D_ZCJump
        BNE     L_DrumZCJumpIRQ

        LDA     R_ChStatus,Y
        AND     #D_AdpcmJumpPcm
        BEQ     L_ExitIrq?
        JMP     L_AdpcmJumpPcmIRQ_D
L_ExitIrq?:
        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl           ;Disable channel IRQ

        LDA     T_ChEnable,Y
        STA     P_SPU_Status            ;Clear INT status
        RTS
;==========================================================================
;ADPCM jump to PCM
;==========================================================================
L_AdpcmJumpPcmIRQ_D:
        LDA     R_ChDrumIndex,Y         ;Get drum index
        TAY
        LDA     T_DrumInstL,Y
        STA     R_InstTabAddrIRQ        ;Instrument table address LB
        LDA     T_DrumInstH,Y
        STA     R_InstTabAddrIRQ+1      ;Instrument table address HB

        LDY     #0BH
        LDA     (R_InstTabAddrIRQ),Y    ;Loop length LB
        STA     R_Ch0LpLenL,X
        INY
        LDA     (R_InstTabAddrIRQ),Y    ;Loop length HB
        STA     R_Ch0LpLenH,X

        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_ZCJump+D_NotePlaying+D_AdpcmJumpPcm+D_NoteOffFlag)
        ORA     #D_DrumPlaying
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     T_ChEnable,Y
        STA     P_SPU_Status
        RTS
;==========================================================================
;Concatenate jump mode
;==========================================================================
L_DrumZCJumpIRQ:
        LDA     R_ChDrumIndex,Y         ;Get drum index
        TAY
        LDA     T_DrumInstL,Y
        STA     R_InstTabAddrIRQ        ;Instrument table address LB
        LDA     T_DrumInstH,Y
        STA     R_InstTabAddrIRQ+1      ;Instrument table address HB

        LDY     #00H
        LDA     (R_InstTabAddrIRQ),Y    ;Get index of sampling rate table
        ASL     A
        TAY
        LDA     T_SampleRateBase,Y
        STA     R_InstAddrIRQ           ;Get low address of sampling rate table
        LDA     T_SampleRateBase+1,Y
        STA     R_InstAddrIRQ+1         ;Get high address of sampling rate table

        LDY     #00H
        LDA     (R_InstAddrIRQ),Y       ;Get sampling rate LB
        STA     R_MIDITempIRQ
        INY
        LDA     (R_InstAddrIRQ),Y       ;Get sampling rate HB
        LDY     R_BackUpChIRQ
        STA     R_Ch0FsH,X

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsH,Y
        .ENDIF

        LDA     R_MIDITempIRQ
        STA     R_Ch0FsL,X

        .IF PITCH_BEND = ON
        STA     R_ChNoteFsL,Y
        .ENDIF

L_ChkDrumModeInZCIrq:
        LDY     #01H
        LDA     (R_InstTabAddrIRQ),Y    ;Get format of CURRENT
        BMI     L_AttackIsPCM_D
L_AttackIsADPCM_D:
        LDY     #08H
        LDA     (R_InstTabAddrIRQ),Y    ;Get format of NEXT
        BPL     L_BothADPCM_ZCIrq_D
        JMP     L_ADPCM_PCM_ZCIrq_D
L_AttackIsPCM_D:
        LDY     #08H
        LDA     (R_InstTabAddrIRQ),Y    ;Get format of NEXT
        BMI     L_BothPCM_ZCIrq_D
;==========================================================================
;Attack(PCM) + Loop(ADPCM)
;==========================================================================
L_PCM_ADPCM_ZCIrq_D:
        LDA     (R_InstTabAddrIRQ),Y    ;Set address of NEXT
        STA     R_Ch0DPTNH,X
        INY
        LDA     (R_InstTabAddrIRQ),Y
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_InstTabAddrIRQ),Y
        STA     R_Ch0DPTNM,X

        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_ZCJump+D_NotePlaying+D_AdpcmJumpPcm+D_NoteOffFlag)
        ORA     #D_DrumPlaying
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     R_Ch0VolCtl,X
        ORA     #D_Bit7
        STA     R_Ch0VolCtl,X

        LDA     R_Ch0VolN,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X
        RTS
;==========================================================================
;Attack(ADPCM) + Loop(ADPCM)
;==========================================================================
L_BothADPCM_ZCIrq_D:
        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_ZCJump+D_NotePlaying+D_AdpcmJumpPcm+D_NoteOffFlag)
        ORA     #D_DrumPlaying
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     T_ChEnable,Y
        STA     P_SPU_Status

        LDA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        STA     R_Ch0VolCtl,X

        LDA     R_Ch0VolN,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X
        RTS
;==========================================================================
;Attack(PCM) + Loop(PCM)
;==========================================================================
L_BothPCM_ZCIrq_D:
        LDY     #0BH
        LDA     (R_InstTabAddrIRQ),Y    ;Get loop length LB
        STA     R_Ch0LpLenL,X
        INY
        LDA     (R_InstTabAddrIRQ),Y    ;Get loop length HB
        STA     R_Ch0LpLenH,X

        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_ZCJump+D_NotePlaying+D_AdpcmJumpPcm+D_NoteOffFlag)
        ORA     #D_DrumPlaying
        STA     R_ChStatus,Y

        LDA     P_SPU_IntCtrl
        AND     T_ChDisable,Y
        STA     P_SPU_IntCtrl

        LDA     T_ChDisable,Y
        STA     P_SPU_Status

        LDA     R_Ch0VolCtl,X
        AND     #~D_Bit7
        STA     R_Ch0VolCtl,X

        LDA     R_Ch0VolN,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X
        RTS
;==========================================================================
;Attack(ADPCM) + Loop(PCM)
;==========================================================================
L_ADPCM_PCM_ZCIrq_D:
        LDA     (R_InstTabAddrIRQ),Y    ;Set address of NEXT
        STA     R_Ch0DPTNH,X
        INY
        LDA     (R_InstTabAddrIRQ),Y
        STA     R_Ch0DPTNL,X
        INY
        LDA     (R_InstTabAddrIRQ),Y
        STA     R_Ch0DPTNM,X

        LDA     R_Ch0VolCtl,X
        ORA     #D_Bit7
        STA     R_Ch0VolCtl,X

        LDA     R_Ch0VolN,X
        AND     #~D_Bit7
        STA     R_Ch0VolN,X

        LDY     R_BackUpChIRQ
        LDA     R_ChStatus,Y
        AND     #.NOT.(D_ZCJump+D_NotePlaying+D_NoteOffFlag)
        ORA     #D_DrumPlaying+D_AdpcmJumpPcm
        STA     R_ChStatus,Y

        LDA     T_ChEnable,Y
        STA     P_SPU_Status
        RTS
;==========================================================================
;Purpose: Select dynamic allocation type
;Input: Y = 0(Dynamic Off), 1(Oldest note out), 2(Newest note out), 3(Minimum volume note out)
;Return: None
;Destroy: A
;==========================================================================
F_SelectDynamic:
        CPY     #00H
        BEQ     L_SetDynamicOff?
        CPY     #01H
        BEQ     L_SetOldestNoteOut?
        CPY     #02H
        BEQ     L_SetNewestNoteOut?
        CPY     #03H
        BEQ     L_SetMinVolNoteOut?
        JMP     L_SetOldestNoteOut?
L_SetDynamicOff?:
        LDA     R_MIDIStatus
        AND     #.NOT.(D_NewestNoteOut+D_MinVolNoteOut)
        ORA     #D_DynaAllocOff
        STA     R_MIDIStatus
        RTS
L_SetOldestNoteOut?:
        LDA     R_MIDIStatus
        AND     #.NOT.(D_DynaAllocOff+D_NewestNoteOut+D_MinVolNoteOut)
        STA     R_MIDIStatus
        RTS
L_SetNewestNoteOut?:
        LDA     R_MIDIStatus
        AND     #.NOT.(D_DynaAllocOff+D_MinVolNoteOut)
        ORA     #D_NewestNoteOut
        STA     R_MIDIStatus
        RTS
L_SetMinVolNoteOut?:
        LDA     R_MIDIStatus
        AND     #.NOT.(D_DynaAllocOff+D_NewestNoteOut)
        ORA     #D_MinVolNoteOut
        STA     R_MIDIStatus
        RTS
;==========================================================================
;Dynamic allocation
;==========================================================================
F_FindOldestNote:                       ;Find oldest note and replace it (Find minimum R_ChTime)
        LDX     R_BackUpCh              ;Check current channel is busy or idle
L_CheckIdleCh?:
        LDA     P_SPU_Enable

        .IF     D_ChannelNo = 4
        AND     #00001111B
        CMP     #00001111B              ;Check if all 4 channels are busy
        .ENDIF

        .IF     D_ChannelNo = 8
        CMP     #11111111B              ;Check if all 8 channels are busy
        .ENDIF

        BEQ     L_FindOldestNote        ;If all channels are busy, find oldest note
        AND     T_ChEnable,X            ;Is current channel busy
        BNE     L_FindNextCh?

        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying        ;Check if the channel is paused in playing speech
        BNE     L_FindNextCh?
        .ENDIF

        STX     R_BackUpCh

        .IF PITCH_BEND = ON
        LDA     #FFH
        STA     R_SpuMidiCh,X
        .ENDIF

        RTS

L_FindNextCh?:
        INX                             ;Find next idle channel
        CPX     #D_ChannelNo
        BCC     L_CheckIdleCh?
        LDX     #00H                    ;Find next idle ch
        JMP     L_CheckIdleCh?

L_FindOldestNote:
        LDX     #D_ChannelNo-1
        LDA     #FFH
        STA     R_MIDITemp
L_SearchLoop?:
        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying
        BNE     L_CheckNextChTime?
        .ENDIF

        LDA     R_SingleNote
        AND     T_ChEnable,X            ;Check if a single note is played
        BNE     L_CheckNextChTime?

        LDA     R_SingleDrum
        AND     T_ChEnable,X            ;Check if a single drum is played
        BNE     L_CheckNextChTime?

        LDA     R_ChStatus,X            ;Check if the channel is in note off
        AND     #D_NoteOffFlag
        BNE     L_OldestNoteOff?

        LDA     R_ChTime,X
        CMP     R_MIDITemp
        BCS     L_CheckNextChTime?
        STA     R_MIDITemp
        STX     R_BackUpCh
L_CheckNextChTime?:
        DEX
        BPL     L_SearchLoop?
        JMP     L_OldestChOff?

L_OldestNoteOff?:
        STX     R_BackUpCh
L_OldestChOff?:
        LDY     R_BackUpCh

        .IF PITCH_BEND = ON
        LDA     #FFH
        STA     R_SpuMidiCh,Y
        .ENDIF

        JSR     F_StopSound             ;Stop enabled channel to prevent the loop region error of original instrument/drum
        RTS
;==========================================================================
F_FindNewestNote:                       ;Find newest note and replace it (Find maximum R_ChTime)
        LDX     R_BackUpCh              ;Check current channel is busy or idle
L_CheckIdleCh?:
        LDA     P_SPU_Enable

        .IF     D_ChannelNo = 4
        AND     #00001111B
        CMP     #00001111B              ;Check if all 4 channels are busy
        .ENDIF

        .IF     D_ChannelNo = 8
        CMP     #11111111B              ;Check if all 8 channels are busy
        .ENDIF

        BEQ     L_FindNewestNote
        AND     T_ChEnable,X            ;Is current channel busy
        BNE     L_FindNextCh?

        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying        ;Check if the channel is paused in playing speech
        BNE     L_FindNextCh?
        .ENDIF

        STX     R_BackUpCh

        .IF PITCH_BEND = ON
        LDA     #FFH
        STA     R_SpuMidiCh,X
        .ENDIF

        RTS

L_FindNextCh?:
        INX                             ;Find next idle channel
        CPX     #D_ChannelNo
        BCC     L_CheckIdleCh?
        LDX     #00H
        JMP     L_CheckIdleCh?

L_FindNewestNote:
        LDX     #D_ChannelNo-1
        LDA     #00H
        STA     R_MIDITemp
L_SearchLoop?:
        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying
        BNE     L_CheckNextChTime?
        .ENDIF

        LDA     R_SingleNote
        AND     T_ChEnable,X            ;Check if a single note is played
        BNE     L_CheckNextChTime?

        LDA     R_SingleDrum
        AND     T_ChEnable,X            ;Check if a single drum is played
        BNE     L_CheckNextChTime?

        LDA     R_ChStatus,X            ;Check if the channel is in note off
        AND     #D_NoteOffFlag
        BNE     L_NewestNoteOff?

        LDA     R_ChTime,X
        CMP     R_MIDITemp
        BCC     L_CheckNextChTime?
        STA     R_MIDITemp
        STX     R_BackUpCh
L_CheckNextChTime?:
        DEX
        BPL     L_SearchLoop?
        JMP     L_NewestChOff?

L_NewestNoteOff?:
        STX     R_BackUpCh
L_NewestChOff?:
        LDY     R_BackUpCh

        .IF PITCH_BEND = ON
        LDA     #FFH
        STA     R_SpuMidiCh,Y
        .ENDIF

        JSR     F_StopSound             ;Stop enabled channel to prevent the loop region error of original instrument/drum
        RTS
;==========================================================================
F_FindMinChVol:                         ;Find a channel of minimum volume and replace it
        LDX     R_BackUpCh              ;Check current channel is busy or idle
L_CheckIdleCh?:
        LDA     P_SPU_Enable

        .IF     D_ChannelNo = 4
        AND     #00001111B
        CMP     #00001111B              ;Check if all 4 channels are busy
        .ENDIF

        .IF     D_ChannelNo = 8
        CMP     #11111111B              ;Check if all 8 channels are busy
        .ENDIF

        BEQ     L_FindMinChVol
        AND     T_ChEnable,X            ;Is current channel busy
        BNE     L_FindNextCh?

        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying        ;Check if the channel is paused in playing speech
        BNE     L_FindNextCh?
        .ENDIF

        STX     R_BackUpCh

        .IF PITCH_BEND = ON
        LDA     #FFH
        STA     R_SpuMidiCh,X
        .ENDIF

        RTS

L_FindNextCh?:
        INX                             ;Find next idle channel
        CPX     #D_ChannelNo
        BCC     L_CheckIdleCh?
        LDX     #00H                    ;Find next idle channel
        JMP     L_CheckIdleCh?

L_FindMinChVol:
        LDA     #FFH
        STA     R_MIDITemp              ;Temporary for volume
        LDX     #D_ChannelNo-1
        STX     R_BackUpCh              ;Temporary for channel
L_SearchLoop?:
        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying
        BNE     L_NextChMinVol?
        .ENDIF

        LDA     R_SingleNote
        AND     T_ChEnable,X            ;Check if a single note is played
        BNE     L_NextChMinVol?

        LDA     R_SingleDrum
        AND     T_ChEnable,X            ;Check if a single drum is played
        BNE     L_NextChMinVol?

        LDA     R_ChStatus,X            ;Check if the channel is in note off
        AND     #D_NoteOffFlag
        BNE     L_MinVolNoteOff?

        LDY     T_ChMap,X
        LDA     R_Ch0VolCtl,Y
        AND     #01111111B
        CMP     R_MIDITemp
        BCS     L_NextChMinVol?
        STA     R_MIDITemp              ;If Volume < R_MIDITemp, replace previous one
        STX     R_BackUpCh              ;Backup channel
L_NextChMinVol?:
        DEX
        BPL     L_SearchLoop?
        JMP     L_MinVolChOff?

L_MinVolNoteOff?:
        STX     R_BackUpCh
L_MinVolChOff?:
        LDY     R_BackUpCh

        .IF PITCH_BEND = ON
        LDA     #FFH
        STA     R_SpuMidiCh,Y
        .ENDIF

        JSR     F_StopSound             ;Stop enabled channel to prevent the loop region error of original instrument/drum
        RTS
;==========================================================================
F_GetMIDIData:
        LDY     #00H
        LDA     R_MIDIDPTR+2            ;Bank
        SEI
        STA     P_Bank
        LDA     (R_MIDIDPTR),Y          ;Fetch data from memory
        CLI
        STA     R_MIDIData

        INC     R_MIDIDPTR
        BNE     L_NotInc?
        INC     R_MIDIDPTR+1
        BNE     L_NotInc?
        INC     R_MIDIDPTR+2
        LDA     #C0H
        STA     R_MIDIDPTR+1
L_NotInc?:
        RTS
;==========================================================================
;Purpose: Pause a MIDI
;Input: None
;Return: None
;Destroy: A, X
;==========================================================================
F_PauseMIDI:
        LDX     #00H
L_HoldLoop:
        .IF SPEECH = ON
        LDA     R_ChSpeechStatus,X
        AND     #D_SpeechPlaying+D_SpeechRepeat+D_SpeechPause
        BNE     L_ChkNextChSp?
        .ENDIF

        SEI
        LDA     P_SPU_Enable
        AND     T_ChDisable,X
        STA     P_SPU_Enable
        CLI

        LDA     #00H
        STA     R_ChTime,X
L_ChkNextChSp?:
        INX
        CPX     #D_ChannelNo
        BCC     L_HoldLoop
        LDA     R_MIDICtrl
        ORA     #D_PauseMIDI
        STA     R_MIDICtrl
L_ExitPauseMIDI?:
        RTS
;==========================================================================
;Purpose: Resume a MIDI which is paused by "F_PauseMIDI"
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_ResumeMIDI:
        LDA     R_MIDICtrl
        AND     #.NOT.(D_PauseMIDI)
        STA     R_MIDICtrl
        RTS
;==========================================================================
;Purpose: Increase tempo while playing a MIDI (Up 6 tempos per step when default = 120)
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_SetTempoUp:
        LDA     R_TempoIndex
        CMP     #D_TempoIndexBegin
        BEQ     L_ExitSetTempoDn
        DEC     R_TempoIndex
L_ExitSetTempoUp:
        RTS
;==========================================================================
;Purpose: Decrease tempo while playing a MIDI (Down 6 tempos per step when default = 120)
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_SetTempoDn:
        LDA     R_TempoIndex
        CMP     #D_TempoIndexEnd
        BEQ     L_ExitSetTempoDn
        INC     R_TempoIndex
L_ExitSetTempoDn:
        RTS
;==========================================================================
;Purpose: Turn off a MIDI
;Input: None
;Return: None
;Destroy: A, X, Y
;==========================================================================
F_MIDIOff:
        LDX     #00H
L_ChkLoop?:     
        LDY     T_ChMap,X
        LDA     R_ChStatus,X
        AND     #D_NotePlaying
        BNE     L_StopCh?
        LDA     R_ChStatus,X
        AND     #D_DrumPlaying
        BEQ     L_ChkNext?
L_StopCh?:
        SEI
        LDA     P_SPU_Enable
        AND     T_ChDisable,X
        STA     P_SPU_Enable
        CLI

        LDA     #00H
        STA     R_ChTime,X
L_ChkNext?:
        INX
        CPX     #D_ChannelNo
        BCC     L_ChkLoop?

        LDA     R_MIDIStatus
        ORA     #D_MusicOff
        AND     #.NOT.(D_MIDIEn)
        STA     R_MIDIStatus
L_ExitMIDIOff?:
        RTS
;==========================================================================
;Purpose: Turn on a MIDI which is turned off by "F_MIDIOff"
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_MIDIOn:
        LDA     R_MIDIStatus
        AND     #.NOT.(D_MusicOff)
        ORA     #D_MIDIEn
        STA     R_MIDIStatus
        RTS
;==========================================================================
;Purpose: Enable MIDI repeat mode (Available index = 0 ~ 127)
;Input: None
;Return: None
;Destroy: A
;==========================================================================
        .IF REPEAT_MIDI = ON
F_RepeatMIDIOn:
        LDA     R_RepeatMIDI
        ORA     #10000000B
        STA     R_RepeatMIDI
        RTS
;==========================================================================
;Purpose: Disable MIDI repeat mode
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_RepeatMIDIOff:
        LDA     R_RepeatMIDI
        AND     #01111111B
        STA     R_RepeatMIDI
        RTS
        .ENDIF
;==========================================================================
;Purpose: Get a user event in a MIDI
;Input: None
;Return: C = 0, no event; C = 1, an event is issued (X = Main index, Y = Sub index)
;Destroy: A, X, Y
;==========================================================================
F_GetUserEvent:
        CLC
        LDA     R_MIDIStatus            ;Check if a user event is detected
        AND     #D_UserEvent
        BEQ     L_ExitGetUserEvent
        LDA     R_MIDIStatus
        AND     #~D_UserEvent           ;Clear flag
        STA     R_MIDIStatus
        SEC
        LDX     R_MainIndex             ;Store main-index to X register
        LDY     R_SubIndex              ;Store sub-index to Y register
L_ExitGetUserEvent:
        RTS
;==========================================================================
;Purpose: Get a MIDI event in a MIDI
;Input: None
;Return: C = 0, no event; C = 1, an event is issued (X = Event pitch)
;Destroy: A, X
;==========================================================================
        .IF MIDI_EVENT = ON
F_GetMIDIEvent:
        CLC
        LDA     R_MIDICtrl              ;Check if a MIDI event is detected
        AND     #D_MIDIEvent
        BEQ     L_ExitGetMIDIEvent
        LDA     R_MIDICtrl
        AND     #~D_MIDIEvent           ;Clear flag
        STA     R_MIDICtrl
        SEC
        LDX     R_MIDIEvent             ;Store MIDI event to X register
L_ExitGetMIDIEvent:
        RTS
        .ENDIF
;==========================================================================
;Purpose: Mute designated MIDI channel
;Input: X = MIDI channel (1 ~ 16)
;Return: None
;Destroy: A
;==========================================================================
        .IF MUTE_MIDI_CH = ON
F_MuteMIDICh:
L_DisMIDICh1?:
        CPX     #01H
        BNE     L_DisMIDICh2?
        LDA     R_EnCh1ToCh8
        ORA     #D_DisMIDICh1
        STA     R_EnCh1ToCh8
L_DisMIDICh2?:
        CPX     #02H
        BNE     L_DisMIDICh3?
        LDA     R_EnCh1ToCh8
        ORA     #D_DisMIDICh2
        STA     R_EnCh1ToCh8
L_DisMIDICh3?:
        CPX     #03H
        BNE     L_DisMIDICh4?
        LDA     R_EnCh1ToCh8
        ORA     #D_DisMIDICh3
        STA     R_EnCh1ToCh8
L_DisMIDICh4?:
        CPX     #04H
        BNE     L_DisMIDICh5?
        LDA     R_EnCh1ToCh8
        ORA     #D_DisMIDICh4
        STA     R_EnCh1ToCh8
L_DisMIDICh5?:
        CPX     #05H
        BNE     L_DisMIDICh6?
        LDA     R_EnCh1ToCh8
        ORA     #D_DisMIDICh5
        STA     R_EnCh1ToCh8
L_DisMIDICh6?:
        CPX     #06H
        BNE     L_DisMIDICh7?
        LDA     R_EnCh1ToCh8
        ORA     #D_DisMIDICh6
        STA     R_EnCh1ToCh8
L_DisMIDICh7?:
        CPX     #07H
        BNE     L_DisMIDICh8?
        LDA     R_EnCh1ToCh8
        ORA     #D_DisMIDICh7
        STA     R_EnCh1ToCh8
L_DisMIDICh8?:
        CPX     #08H
        BNE     L_DisMIDICh9?
        LDA     R_EnCh1ToCh8
        ORA     #D_DisMIDICh8
        STA     R_EnCh1ToCh8
L_DisMIDICh9?:
        CPX     #09H
        BNE     L_DisMIDICh10?
        LDA     R_EnCh9ToCh16
        ORA     #D_DisMIDICh9
        STA     R_EnCh9ToCh16
L_DisMIDICh10?:
        CPX     #0AH
        BNE     L_DisMIDICh11?
        LDA     R_EnCh9ToCh16
        ORA     #D_DisMIDICh10
        STA     R_EnCh9ToCh16
L_DisMIDICh11?:
        CPX     #0BH
        BNE     L_DisMIDICh12?
        LDA     R_EnCh9ToCh16
        ORA     #D_DisMIDICh11
        STA     R_EnCh9ToCh16
L_DisMIDICh12?:
        CPX     #0CH
        BNE     L_DisMIDICh13?
        LDA     R_EnCh9ToCh16
        ORA     #D_DisMIDICh12
        STA     R_EnCh9ToCh16
L_DisMIDICh13?:
        CPX     #0DH
        BNE     L_DisMIDICh14?
        LDA     R_EnCh9ToCh16
        ORA     #D_DisMIDICh13
        STA     R_EnCh9ToCh16
L_DisMIDICh14?:
        CPX     #0EH
        BNE     L_DisMIDICh15?
        LDA     R_EnCh9ToCh16
        ORA     #D_DisMIDICh14
        STA     R_EnCh9ToCh16
L_DisMIDICh15?:
        CPX     #0FH
        BNE     L_DisMIDICh16?
        LDA     R_EnCh9ToCh16
        ORA     #D_DisMIDICh15
        STA     R_EnCh9ToCh16
L_DisMIDICh16?:
        CPX     #10H
        BNE     L_ExitMuteMIDICh
        LDA     R_EnCh9ToCh16
        ORA     #D_DisMIDICh16
        STA     R_EnCh9ToCh16
L_ExitMuteMIDICh:
        RTS
;==========================================================================
;Purpose: Restore the MIDI channel which is muted by "F_MuteMIDICh"
;Input: X = MIDI channel (1 ~ 16)
;Return: None
;Destroy: A
;==========================================================================
F_UnMuteMIDICh:
L_DisMIDICh1?:
        CPX     #01H
        BNE     L_DisMIDICh2?
        LDA     R_EnCh1ToCh8
        AND     #.NOT.(D_DisMIDICh1)
        STA     R_EnCh1ToCh8
L_DisMIDICh2?:
        CPX     #02H
        BNE     L_DisMIDICh3?
        LDA     R_EnCh1ToCh8
        AND     #.NOT.(D_DisMIDICh2)
        STA     R_EnCh1ToCh8
L_DisMIDICh3?:
        CPX     #03H
        BNE     L_DisMIDICh4?
        LDA     R_EnCh1ToCh8
        AND     #.NOT.(D_DisMIDICh3)
        STA     R_EnCh1ToCh8
L_DisMIDICh4?:
        CPX     #04H
        BNE     L_DisMIDICh5?
        LDA     R_EnCh1ToCh8
        AND     #.NOT.(D_DisMIDICh4)
        STA     R_EnCh1ToCh8
L_DisMIDICh5?:
        CPX     #05H
        BNE     L_DisMIDICh6?
        LDA     R_EnCh1ToCh8
        AND     #.NOT.(D_DisMIDICh5)
        STA     R_EnCh1ToCh8
L_DisMIDICh6?:
        CPX     #06H
        BNE     L_DisMIDICh7?
        LDA     R_EnCh1ToCh8
        AND     #.NOT.(D_DisMIDICh6)
        STA     R_EnCh1ToCh8
L_DisMIDICh7?:
        CPX     #07H
        BNE     L_DisMIDICh8?
        LDA     R_EnCh1ToCh8
        AND     #.NOT.(D_DisMIDICh7)
        STA     R_EnCh1ToCh8
L_DisMIDICh8?:
        CPX     #08H
        BNE     L_DisMIDICh9?
        LDA     R_EnCh1ToCh8
        AND     #.NOT.(D_DisMIDICh8)
        STA     R_EnCh1ToCh8
L_DisMIDICh9?:
        CPX     #09H
        BNE     L_DisMIDICh10?
        LDA     R_EnCh9ToCh16
        AND     #.NOT.(D_DisMIDICh9)
        STA     R_EnCh9ToCh16
L_DisMIDICh10?:
        CPX     #0AH
        BNE     L_DisMIDICh11?
        LDA     R_EnCh9ToCh16
        AND     #.NOT.(D_DisMIDICh10)
        STA     R_EnCh9ToCh16
L_DisMIDICh11?:
        CPX     #0BH
        BNE     L_DisMIDICh12?
        LDA     R_EnCh9ToCh16
        AND     #.NOT.(D_DisMIDICh11)
        STA     R_EnCh9ToCh16
L_DisMIDICh12?:
        CPX     #0CH
        BNE     L_DisMIDICh13?
        LDA     R_EnCh9ToCh16
        AND     #.NOT.(D_DisMIDICh12)
        STA     R_EnCh9ToCh16
L_DisMIDICh13?:
        CPX     #0DH
        BNE     L_DisMIDICh14?
        LDA     R_EnCh9ToCh16
        AND     #.NOT.(D_DisMIDICh13)
        STA     R_EnCh9ToCh16
L_DisMIDICh14?:
        CPX     #0EH
        BNE     L_DisMIDICh15?
        LDA     R_EnCh9ToCh16
        AND     #.NOT.(D_DisMIDICh14)
        STA     R_EnCh9ToCh16
L_DisMIDICh15?:
        CPX     #0FH
        BNE     L_DisMIDICh16?
        LDA     R_EnCh9ToCh16
        AND     #.NOT.(D_DisMIDICh15)
        STA     R_EnCh9ToCh16
L_DisMIDICh16?:
        CPX     #10H
        BNE     L_ExitUnMuteMIDICh
        LDA     R_EnCh9ToCh16
        AND     #.NOT.(D_DisMIDICh16)
        STA     R_EnCh9ToCh16
L_ExitUnMuteMIDICh:
        RTS
        .ENDIF
;==========================================================================
;Purpose: Shift MIDI key up with one semitone per step
;Input: None
;Return: None
;Destroy: A
;==========================================================================
        .IF MIDI_KEY_SHIFT = ON
F_KeyShiftUp:
        CLC
        LDA     R_KeyShift
        ADC     #01H
        STA     R_KeyShift
L_ExitKeyShiftUp:
        RTS
;==========================================================================
;Purpose: Shift MIDI key down with one semitone per step
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_KeyShiftDn:
        SEC
        LDA     R_KeyShift
        SBC     #01H
        STA     R_KeyShift
L_ExitKeyShiftDn:
        RTS
        .ENDIF
;==========================================================================
;Purpose: Change the instrument of designated MIDI channel
;Input: A = New instrument index; X = MIDI channel (1 ~ 16)
;Return: None
;Destroy: X
;==========================================================================
F_ChangeInst:
        CPX     #00H                    ;Check if channel = 0
        BEQ     L_ExitChangeInst
        CPX     #11H                    ;Check if channel > 16
        BCS     L_ExitChangeInst
        DEX
        STA     R_ChInst,X
L_ExitChangeInst:
        RTS
;==========================================================================
;Purpose: Enable volume control of designated MIDI channel
;Input: X = MIDI channel (1 ~ 16)
;Return: None
;Destroy: A
;==========================================================================
        .IF CTRL_MIDI_CH_VOL = ON
F_EnMIDIChVolCtrl:
        CPX     #01H
        BNE     L_CtrlCh2Vol?
        LDA     R_CtrlVolCh1ToCh8
        ORA     #D_CtrlCh1Vol
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh2Vol?:
        CPX     #02H
        BNE     L_CtrlCh3Vol?
        LDA     R_CtrlVolCh1ToCh8
        ORA     #D_CtrlCh2Vol
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh3Vol?:
        CPX     #03H
        BNE     L_CtrlCh4Vol?
        LDA     R_CtrlVolCh1ToCh8
        ORA     #D_CtrlCh3Vol
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh4Vol?:
        CPX     #04H
        BNE     L_CtrlCh5Vol?
        LDA     R_CtrlVolCh1ToCh8
        ORA     #D_CtrlCh4Vol
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh5Vol?:
        CPX     #05H
        BNE     L_CtrlCh6Vol?
        LDA     R_CtrlVolCh1ToCh8
        ORA     #D_CtrlCh5Vol
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh6Vol?:
        CPX     #06H
        BNE     L_CtrlCh7Vol?
        LDA     R_CtrlVolCh1ToCh8
        ORA     #D_CtrlCh6Vol
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh7Vol?:
        CPX     #07H
        BNE     L_CtrlCh8Vol?
        LDA     R_CtrlVolCh1ToCh8
        ORA     #D_CtrlCh7Vol
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh8Vol?:
        CPX     #08H
        BNE     L_CtrlCh9Vol?
        LDA     R_CtrlVolCh1ToCh8
        ORA     #D_CtrlCh8Vol
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh9Vol?:
        CPX     #09H
        BNE     L_CtrlCh10Vol?
        LDA     R_CtrlVolCh9ToCh16
        ORA     #D_CtrlCh9Vol
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh10Vol?:
        CPX     #0AH
        BNE     L_CtrlCh11Vol?
        LDA     R_CtrlVolCh9ToCh16
        ORA     #D_CtrlCh10Vol
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh11Vol?:
        CPX     #0BH
        BNE     L_CtrlCh12Vol?
        LDA     R_CtrlVolCh9ToCh16
        ORA     #D_CtrlCh11Vol
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh12Vol?:
        CPX     #0CH
        BNE     L_CtrlCh13Vol?
        LDA     R_CtrlVolCh9ToCh16
        ORA     #D_CtrlCh12Vol
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh13Vol?:
        CPX     #0DH
        BNE     L_CtrlCh14Vol?
        LDA     R_CtrlVolCh9ToCh16
        ORA     #D_CtrlCh13Vol
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh14Vol?:
        CPX     #0EH
        BNE     L_CtrlCh15Vol?
        LDA     R_CtrlVolCh9ToCh16
        ORA     #D_CtrlCh14Vol
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh15Vol?:
        CPX     #0FH
        BNE     L_CtrlCh16Vol?
        LDA     R_CtrlVolCh9ToCh16
        ORA     #D_CtrlCh15Vol
        STA     R_CtrlVolCh9ToCh16
L_CtrlCh16Vol?:
        CPX     #10H
        BNE     L_ExitEnMIDIChVolCtrl
        LDA     R_CtrlVolCh9ToCh16
        ORA     #D_CtrlCh16Vol
        STA     R_CtrlVolCh9ToCh16
L_ExitEnMIDIChVolCtrl:
        RTS
;==========================================================================
;Purpose: Disable volume control of designated MIDI channel
;Input: X = MIDI channel (1 ~ 16)
;Return: None
;Destroy: A
;==========================================================================
F_DisMIDIChVolCtrl:
        CPX     #01H
        BNE     L_CtrlCh2Vol?
        LDA     R_CtrlVolCh1ToCh8
        AND     #.NOT.(D_CtrlCh1Vol)
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh2Vol?:
        CPX     #02H
        BNE     L_CtrlCh3Vol?
        LDA     R_CtrlVolCh1ToCh8
        AND     #.NOT.(D_CtrlCh2Vol)
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh3Vol?:
        CPX     #03H
        BNE     L_CtrlCh4Vol?
        LDA     R_CtrlVolCh1ToCh8
        AND     #.NOT.(D_CtrlCh3Vol)
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh4Vol?:
        CPX     #04H
        BNE     L_CtrlCh5Vol?
        LDA     R_CtrlVolCh1ToCh8
        AND     #.NOT.(D_CtrlCh4Vol)
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh5Vol?:
        CPX     #05H
        BNE     L_CtrlCh6Vol?
        LDA     R_CtrlVolCh1ToCh8
        AND     #.NOT.(D_CtrlCh5Vol)
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh6Vol?:
        CPX     #06H
        BNE     L_CtrlCh7Vol?
        LDA     R_CtrlVolCh1ToCh8
        AND     #.NOT.(D_CtrlCh6Vol)
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh7Vol?:
        CPX     #07H
        BNE     L_CtrlCh8Vol?
        LDA     R_CtrlVolCh1ToCh8
        AND     #.NOT.(D_CtrlCh7Vol)
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh8Vol?:
        CPX     #08H
        BNE     L_CtrlCh9Vol?
        LDA     R_CtrlVolCh1ToCh8
        AND     #.NOT.(D_CtrlCh8Vol)
        STA     R_CtrlVolCh1ToCh8
        RTS
L_CtrlCh9Vol?:
        CPX     #09H
        BNE     L_CtrlCh10Vol?
        LDA     R_CtrlVolCh9ToCh16
        AND     #.NOT.(D_CtrlCh9Vol)
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh10Vol?:
        CPX     #0AH
        BNE     L_CtrlCh11Vol?
        LDA     R_CtrlVolCh9ToCh16
        AND     #.NOT.(D_CtrlCh10Vol)
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh11Vol?:
        CPX     #0BH
        BNE     L_CtrlCh12Vol?
        LDA     R_CtrlVolCh9ToCh16
        AND     #.NOT.(D_CtrlCh11Vol)
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh12Vol?:
        CPX     #0CH
        BNE     L_CtrlCh13Vol?
        LDA     R_CtrlVolCh9ToCh16
        AND     #.NOT.(D_CtrlCh12Vol)
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh13Vol?:
        CPX     #0DH
        BNE     L_CtrlCh14Vol?
        LDA     R_CtrlVolCh9ToCh16
        AND     #.NOT.(D_CtrlCh13Vol)
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh14Vol?:
        CPX     #0EH
        BNE     L_CtrlCh15Vol?
        LDA     R_CtrlVolCh9ToCh16
        AND     #.NOT.(D_CtrlCh14Vol)
        STA     R_CtrlVolCh9ToCh16
        RTS
L_CtrlCh15Vol?:
        CPX     #0FH
        BNE     L_CtrlCh16Vol?
        LDA     R_CtrlVolCh9ToCh16
        AND     #.NOT.(D_CtrlCh15Vol)
        STA     R_CtrlVolCh9ToCh16
L_CtrlCh16Vol?:
        CPX     #10H
        BNE     L_ExitDisMIDIChVolCtrl
        LDA     R_CtrlVolCh9ToCh16
        AND     #.NOT.(D_CtrlCh16Vol)
        STA     R_CtrlVolCh9ToCh16
L_ExitDisMIDIChVolCtrl:
        RTS
;==========================================================================
;Purpose: Enable volume control of all MIDI channels
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_EnAllMIDIChVolCtrl:
        LDA     #FFH
        STA     R_CtrlVolCh1ToCh8
        STA     R_CtrlVolCh9ToCh16
        RTS
;==========================================================================
;Purpose: Disable volume control of all MIDI channels
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_DisAllMIDIChVolCtrl:
        LDA     #00H
        STA     R_CtrlVolCh1ToCh8
        STA     R_CtrlVolCh9ToCh16
        RTS
;==========================================================================
;Purpose: Set volume up of designated MIDI channel
;Description: Increment = 1/10 per step, it can be up to 20 levels
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_SetMIDIChVolUp:
        SEC
        LDA     R_CtrlMIDIChVol
        SBC     #02H
        BMI     L_SetToMaxLevel?
        STA     R_CtrlMIDIChVol
        RTS
L_SetToMaxLevel?:
        LDA     #00H
        STA     R_CtrlMIDIChVol
L_ExitSetMIDIChVolUp:
        RTS
;==========================================================================
;Purpose: Set volume down of designated MIDI channel
;Description: Decrement = 1/10 per step, it can be down to 10 levels
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_SetMIDIChVolDn:
        LDA     R_CtrlMIDIChVol
        CLC
        ADC     #02H
        CMP     #D_MinVolumeLevel
        BCS     L_SetToMinLevel?
        STA     R_CtrlMIDIChVol
        RTS
L_SetToMinLevel?:
        LDA     #D_MinVolumeLevel-2
        STA     R_CtrlMIDIChVol
L_ExitSetMIDIChVolDn:
        RTS
;==========================================================================
;Purpose: Deal with the volume control of designated MIDI channel
;Description: Used for kernel module
;Input: X = Current channel number 
;Return A = New volume = R_VeloVol,X
;Destroy: A, X, Y    
;==========================================================================
F_CheckMIDIChVol:
        LDA     R_InstIndex,X
        TAY                             ;Y = MIDI Channel
L_CtrlCh1Vol?:
        LDA     R_CtrlVolCh1ToCh8
        AND     #D_CtrlCh1Vol
        BEQ     L_CtrlCh2Vol?
        CPY     #00H
        BNE     L_CtrlCh2Vol?
        JMP     L_SetVolLevel?
L_CtrlCh2Vol?:
        LDA     R_CtrlVolCh1ToCh8
        AND     #D_CtrlCh2Vol
        BEQ     L_CtrlCh3Vol?
        CPY     #01H
        BNE     L_CtrlCh3Vol?
        JMP     L_SetVolLevel?
L_CtrlCh3Vol?:
        LDA     R_CtrlVolCh1ToCh8
        AND     #D_CtrlCh3Vol
        BEQ     L_CtrlCh4Vol?
        CPY     #02H
        BNE     L_CtrlCh4Vol?
        JMP     L_SetVolLevel?
L_CtrlCh4Vol?:
        LDA     R_CtrlVolCh1ToCh8
        AND     #D_CtrlCh4Vol
        BEQ     L_CtrlCh5Vol?
        CPY     #03H
        BNE     L_CtrlCh5Vol?
        JMP     L_SetVolLevel?
L_CtrlCh5Vol?:
        LDA     R_CtrlVolCh1ToCh8
        AND     #D_CtrlCh5Vol
        BEQ     L_CtrlCh6Vol?
        CPY     #04H
        BNE     L_CtrlCh6Vol?
        JMP     L_SetVolLevel?
L_CtrlCh6Vol?:
        LDA     R_CtrlVolCh1ToCh8
        AND     #D_CtrlCh6Vol
        BEQ     L_CtrlCh7Vol?
        CPY     #05H
        BNE     L_CtrlCh7Vol?
        JMP     L_SetVolLevel?
L_CtrlCh7Vol?:
        LDA     R_CtrlVolCh1ToCh8
        AND     #D_CtrlCh7Vol
        BEQ     L_CtrlCh8Vol?
        CPY     #06H
        BNE     L_CtrlCh8Vol?
        JMP     L_SetVolLevel?
L_CtrlCh8Vol?:
        LDA     R_CtrlVolCh1ToCh8
        AND     #D_CtrlCh8Vol
        BEQ     L_CtrlCh9Vol?
        CPY     #07H
        BNE     L_CtrlCh9Vol?
        JMP     L_SetVolLevel?
L_CtrlCh9Vol?:
        LDA     R_CtrlVolCh9ToCh16
        AND     #D_CtrlCh9Vol
        BEQ     L_CtrlCh11Vol?
        CPY     #08H
        BNE     L_CtrlCh11Vol?
        JMP     L_SetVolLevel?
L_CtrlCh11Vol?:
        LDA     R_CtrlVolCh9ToCh16
        AND     #D_CtrlCh11Vol
        BEQ     L_CtrlCh12Vol?
        CPY     #0AH
        BNE     L_CtrlCh12Vol?
        JMP     L_SetVolLevel?
L_CtrlCh12Vol?:
        LDA     R_CtrlVolCh9ToCh16
        AND     #D_CtrlCh12Vol
        BEQ     L_CtrlCh13Vol?
        CPY     #0BH
        BNE     L_CtrlCh13Vol?
        JMP     L_SetVolLevel?
L_CtrlCh13Vol?:
        LDA     R_CtrlVolCh9ToCh16
        AND     #D_CtrlCh13Vol
        BEQ     L_CtrlCh14Vol?
        CPY     #0CH
        BNE     L_CtrlCh14Vol?
        JMP     L_SetVolLevel?
L_CtrlCh14Vol?:
        LDA     R_CtrlVolCh9ToCh16
        AND     #D_CtrlCh14Vol
        BEQ     L_CtrlCh15Vol?
        CPY     #0DH
        BNE     L_CtrlCh15Vol?
        JMP     L_SetVolLevel?
L_CtrlCh15Vol?:
        LDA     R_CtrlVolCh9ToCh16
        AND     #D_CtrlCh15Vol
        BEQ     L_CtrlCh16Vol?
        CPY     #0EH
        BNE     L_CtrlCh16Vol?
        JMP     L_SetVolLevel?
L_CtrlCh16Vol?:
        LDA     R_CtrlVolCh9ToCh16
        AND     #D_CtrlCh16Vol
        BEQ     L_ExitCheckMIDIChVol
        CPY     #0FH
        BNE     L_ExitCheckMIDIChVol
L_SetVolLevel?:
        LDA     R_VeloVol,X
        STA     R_FaciendL
        LDX     R_CtrlMIDIChVol
        LDA     T_VolumeLevel,X
        STA     R_MultiplierL
        INX
        LDA     T_VolumeLevel,X
        STA     R_MultiplierM
        %Multi8X16      R_FaciendL,R_MultiplierL,R_MultiplierM

        LDX     R_BackUpCh
        LDA     R_ProductH              ;Overflow check
        BNE     L_SetToMax?
        LDA     R_ProductM
        CMP     #80H
        BCS     L_SetToMax?
        STA     R_VeloVol,X
        RTS
L_SetToMax?:
        LDA     #7FH
        STA     R_VeloVol,X
L_ExitCheckMIDIChVol:
        RTS
        .ENDIF
;==========================================================================
;Purpose: Pitch bend up 39/10000 times per step, it can be up to 2 semitones at most
;Input: None
;Return: None
;Destroy: A
;==========================================================================
        .IF PITCH_BEND = ON
F_SetPitchBendUp:
        SEC
        LDA     R_PitchBendLevel
        SBC     #02H
        BMI     L_SetToMaxPBLevel?
        STA     R_PitchBendLevel
        RTS
L_SetToMaxPBLevel?:
        LDA     #00H
        STA     R_PitchBendLevel
L_ExitSetPitchBendUp:
        RTS
;==========================================================================
;Purpose: Pitch bend down 39/10000 times per step, it can be down to 2 semitones at most
;Input: None
;Return: None
;Destroy: A
;==========================================================================
F_SetPitchBendDn:
        CLC
        LDA     R_PitchBendLevel
        ADC     #02H
        CMP     #D_EndPBLevel
        BCS     L_SetToMinPBLevel?
        STA     R_PitchBendLevel
        RTS
L_SetToMinPBLevel?:
        LDA     #D_EndPBLevel-2
        STA     R_PitchBendLevel
L_ExitSetPitchBendDn:
        RTS
;==========================================================================
;Purpose: Tone pitch bend
;Description: Modify the hardware frequency to generate the pitch bend effect according to the pitch bend level
;Pitch Bend Level: 04H = Down two semitones
;                  20H = Base pitch
;                  3FH = Up two semitones
;Input: X = Channel number, R_PitchBendLevel = Pitch bend Level
;Return: None
;Destroy: A, X, Y
;==========================================================================
F_TonePitchBend:
        LDA     R_ChNoteFsL,X           ;Get frequency low byte of pitch bend note
        STA     R_FaciendL
        LDA     R_ChNoteFsH,X           ;Get frequency high byte of pitch bend note
        STA     R_FaciendM
        LDY     T_ChMap,X
        SEC
        LDA     #3FH
        SBC     R_PitchBendLevel
        ASL     A
        TAX
        LDA     T_PitchBend,X
        STA     R_MultiplierL
        INX
        LDA     T_PitchBend,X
        STA     R_MultiplierM
        %Multi16X16     R_FaciendL,R_FaciendM,R_MultiplierL,R_MultiplierM

        LDA     R_ProductH
        LDX     R_ProductM
        SEI
        STA     R_Ch0FsH,Y
        STX     R_Ch0FsL,Y
        CLI
        RTS
        .ENDIF
;==========================================================================
;Purpose: Play a single note from instrument library (*.lib)
;Input: R_BackUpCh = SPU channel (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;       R_InstIndex = MIDI channel (0 ~ 15)
;       R_Note = Note pitch
;       R_ChInst = Instrument index
;       R_VeloVol = Volume X Velocity
;Return: None
;Destroy: A, X, Y
;Note: To prevent the instrument of a single note is changed while playing a MIDI.
;      The MIDI channel which is used for a single note should be reserved in the MIDI.
;==========================================================================
F_PlaySingleNote:
        LDA     R_BackUpCh
        CMP     #D_ChannelNo
        BCS     L_ExitPlaySingleNote
        SEI
        LDA     R_ChStatus,X
        AND     #.NOT.(D_NoteOffFlag)
        STA     R_ChStatus,X
        CLI

        LDA     P_INT_CtrlH             ;Enable SPU IRQ
        ORA     #D_SPUIntEn
        STA     P_INT_CtrlH

        LDA     R_SingleNote
        ORA     T_ChEnable,X            ;Set flag
        STA     R_SingleNote

        LDA     R_SingleDrum
        AND     T_ChDisable,X           ;If a single drum is played, clear it
        STA     R_SingleDrum

        JSR     F_InitialADSR
        JSR     F_ChPlayTone
        %TurnOnDAC                      ;Before playing a single note, the push-pull DAC must be turned on
L_ExitPlaySingleNote:
        RTS
;==========================================================================
;Example for playing a single note with CH3
;==========================================================================
;        LDX     #03H
;        STX     R_BackUpCh              ;Play a single note with SPU CH3
;        TXA
;        STA     R_InstIndex,X           ;MIDI channel of current SPU channel

;        LDA     #45H
;        STA     R_Note,X                ;Note pitch

;        LDA     #04H
;        STA     R_ChInst,X              ;Instrument index

;        LDA     #7FH
;        ASL     A
;        STA     P_MulF
;        LDA     #7FH
;        STA     P_MulM
;        LDA     #D_MulAct
;        STA     P_MulAct
;L_Wait?:
;        BIT     P_MulAct
;        BNE     L_Wait?
;        LDA     P_MulOutH
;        LDX     R_BackUpCh
;        STA     R_VeloVol,X             ;Volume X Velocity
;        JSR     F_PlaySingleNote        ;Play a single note
;==========================================================================
;Purpose: Play a single drum from .lib file(Instrument Library)
;Input: R_BackUpCh = SPU channel (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;       R_DrumIndex = Drum index
;       R_VeloVol = Volume X Velocity
;Return: None
;Destroy: A, X, Y
;==========================================================================
F_PlaySingleDrum:
        LDA     R_BackUpCh
        CMP     #D_ChannelNo
        BCS     L_ExitPlaySingleDrum
        SEI
        LDA     R_ChStatus,X
        AND     #.NOT.(D_NoteOffFlag)
        STA     R_ChStatus,X
        CLI

        LDA     P_INT_CtrlH             ;Enable SPU IRQ
        ORA     #D_SPUIntEn
        STA     P_INT_CtrlH

        LDA     R_SingleDrum
        ORA     T_ChEnable,X            ;Set flag
        STA     R_SingleDrum

        LDA     R_SingleNote
        AND     T_ChDisable,X           ;If a single note is played, clear it
        STA     R_SingleNote

        JSR     F_InitialDrumADSR
        JSR     F_PlayDrum
        %TurnOnDAC                      ;Before playing a single drum, the push-pull DAC must be turned on
L_ExitPlaySingleDrum:
        RTS
;==========================================================================
;Example for playing a single drum with CH2
;==========================================================================
;        LDX     #02H
;        STX     R_BackUpCh              ;Play a single drum with SPU CH2
;        LDA     #03H
;        STA     R_DrumIndex             ;Drum index

;        LDA     #7FH
;        ASL     A
;        STA     P_MulF
;        LDA     #7FH
;        STA     P_MulM
;        LDA     #D_MulAct
;        STA     P_MulAct
;L_Wait?:
;        BIT     P_MulAct
;        BNE     L_Wait?
;        LDA     P_MulOutH
;        LDX     R_BackUpCh
;        STA     R_VeloVol,X             ;Volume X Velocity
;        JSR     F_PlaySingleDrum        ;Play a single drum
;==========================================================================
;Purpose: Release a single note/drum on designated SPU channel
;Input: X = Channel number (GPCD6: 0 ~ 3; GPCD9: 0 ~ 7)
;Return: None
;Destroy: A, X
;==========================================================================
F_SingleNoteOff:
        CPX     #D_ChannelNo
        BCS     L_ExitSingleNoteOff
        SEI
        LDA     R_ChStatus,X            ;Enable note off release function
        ORA     #D_NoteOffFlag
        STA     R_ChStatus,X
        CLI

        LDA     #00H
        STA     R_VolumeFloat,X         ;Initial float of volume
L_ExitSingleNoteOff:
        RTS
;==========================================================================
